/*
 * Copyright (c) 1997 Picture Elements, Inc.
 *    Stephen Williams (steve@picturel.com)
 *
 *    c/o Picture Elements, Inc.
 *    777 Panoramic Way
 *    Berkeley, CA, 94704
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
#ident "$Id: pciconf.c,v 1.1 1997/05/21 03:04:35 steve Exp $"
/*
 * $Log: pciconf.c,v $
 * Revision 1.1  1997/05/21 03:04:35  steve
 *  Add support for Windows NT.
 *
 */

# include  <ntddk.h>
# include  "pciconf.h"
# include  "pciconfx.h"

# define SPACE_LIMIT 256
# define FILE_DEVICE_PCICONF FILE_DEVICE_UNKNOWN

# define PCICONF "pciconf"

struct Instance {

	/* These values represent a device that the process is
	   interested in. The device may or may not exist. */

      struct pciconf_info info;

	/* These are details about the device I found that matches the
	   above parameters. Use these to actually manipulate the
	   device of interest. these values are useable if the
	   valid_flag it true. */

      unsigned char bus;
      PCI_SLOT_NUMBER slot;

      int valid_flag;
};


/*
 * Use the device information to locate the board/function and set the
 * valid_flag if the information exists.
 */
static void match_device(struct Instance*xsp)
{
      PCI_COMMON_CONFIG config;
      PCI_SLOT_NUMBER slot;
      unsigned index = xsp->info.instance;
      unsigned bus, dev, fn;

      slot.u.AsULONG = 0;
      for (bus = 0 ;  bus < 256 ;  bus += 1)
	    for (dev = 0 ;  dev < 32 ;  dev += 1)
		  for (fn = 0 ;  fn < 8 ;  fn += 1) {

			slot.u.bits.DeviceNumber = dev;
			slot.u.bits.FunctionNumber = fn;

			switch (HalGetBusData(PCIConfiguration, bus,
					      slot.u.AsULONG, &config, 4)) {
			    case 0:
			      xsp->valid_flag = 0;
			      return;

			    case 2:
			      fn = 8;
			      break;

			    default:
			      if ((config.VendorID == xsp->info.vendor_id)
				  &&(config.DeviceID ==xsp->info.device_id))
				    if (index == 0) {
					  xsp->bus = bus;
					  xsp->slot = slot;
					  xsp->valid_flag = 1;
					  return;
				    } else {
					  index -= 1;
				    }
			      break;
			}
		  }

      xsp->valid_flag = 0;
}

/*
 * xxcreate allocates an instance structure and attaches it to the file
 * object for the process opening me. This gives the process the
 * impression that every open leads to a unique device.
 *
 * xxclose() releases the allocated structure.
 */

static NTSTATUS xxcreate(DEVICE_OBJECT*dev, IRP*irp)
{
      IO_STACK_LOCATION *irpstack = IoGetCurrentIrpStackLocation(irp);
      FILE_OBJECT *file = irpstack->FileObject;
      struct Instance *xsp;

      xsp = (struct Instance*)ExAllocatePool(NonPagedPool,
					     sizeof(struct Instance));
      if (xsp == 0) return STATUS_INSUFFICIENT_RESOURCES;

      xsp->info.vendor_id = 0;
      xsp->info.device_id = 0;
      xsp->info.instance = 0;
      xsp->valid_flag = 0;

      file->FsContext = xsp;

      irp->IoStatus.Status = STATUS_SUCCESS;
      irp->IoStatus.Information = 0;
      IoCompleteRequest(irp, IO_NO_INCREMENT);

      return STATUS_SUCCESS;
}

static NTSTATUS xxclose(DEVICE_OBJECT*dev, IRP*irp)
{
      IO_STACK_LOCATION *irpstack = IoGetCurrentIrpStackLocation(irp);
      FILE_OBJECT *file = irpstack->FileObject;
      struct Instance *xsp = (struct Instance*)file->FsContext;

      file->FsContext = 0;
      ExFreePool(xsp);

      irp->IoStatus.Status = STATUS_SUCCESS;
      irp->IoStatus.Information = 0;
      IoCompleteRequest(irp, IO_NO_INCREMENT);

      return STATUS_SUCCESS;
}


static NTSTATUS xxread(DEVICE_OBJECT*dev, IRP*irp)
{
      IO_STACK_LOCATION *irpstack = IoGetCurrentIrpStackLocation(irp);
      FILE_OBJECT *file = irpstack->FileObject;
      struct Instance *xsp = (struct Instance*)file->FsContext;

      char*bytes = irp->AssociatedIrp.SystemBuffer;
      int tcount = irpstack->Parameters.Read.Length;
      unsigned long offset = irpstack->Parameters.Read.ByteOffset.LowPart;

      unsigned readbase, readcnt;
      char buffer[SPACE_LIMIT];

      unsigned long rc;

      if (! xsp->valid_flag) {
	    irp->IoStatus.Status = STATUS_UNSUCCESSFUL;
	    irp->IoStatus.Information = 0;
	    IoCompleteRequest(irp, IO_NO_INCREMENT);
	    return STATUS_SUCCESS;
      }

      if (offset >= SPACE_LIMIT) {
	    irp->IoStatus.Status = STATUS_UNSUCCESSFUL;
	    irp->IoStatus.Information = 0;
	    IoCompleteRequest(irp, IO_NO_INCREMENT);
	    return STATUS_SUCCESS;
      }

      if ((offset + tcount) > SPACE_LIMIT)
	    tcount = SPACE_LIMIT - offset;

	/* The PCI configuration space is by definition only 256
	   bytes, so it is reasonable for me to simply copy all at
	   once what the customer wants into a local buffer. Align
	   both ends to DWORD boundries because some motherboard
	   chipsets have difficulty with reading smaller then DWORD
	   accesses from the configuration space. */

      readbase = offset & 0xfc;
      readcnt  = tcount + offset - readbase;
      readcnt = (readcnt + 0x03) & 0xfc;

      rc = HalGetBusDataByOffset(PCIConfiguration, xsp->bus,
				 xsp->slot.u.AsULONG, buffer,
				 readbase, readcnt);


	/* Now, just copy the interesting part without the rounding
	   extra into the original caller's buffer. */

      memcpy(bytes, buffer + offset - readbase, tcount);


      irp->IoStatus.Status = STATUS_SUCCESS;
      irp->IoStatus.Information = tcount;
      IoCompleteRequest(irp, IO_NO_INCREMENT);

      return STATUS_SUCCESS;
}


static NTSTATUS xxwrite(DEVICE_OBJECT*dev, IRP*irp)
{
      IO_STACK_LOCATION *irpstack = IoGetCurrentIrpStackLocation(irp);
      FILE_OBJECT *file = irpstack->FileObject;
      struct Instance *xsp = (struct Instance*)file->FsContext;

      const char*bytes = irp->AssociatedIrp.SystemBuffer;
      int tcount = irpstack->Parameters.Write.Length;
      unsigned long offset = irpstack->Parameters.Write.ByteOffset.LowPart;

      if ((offset + tcount) > SPACE_LIMIT)
	    tcount = SPACE_LIMIT - offset;

      if (! xsp->valid_flag) {
	    irp->IoStatus.Status = STATUS_UNSUCCESSFUL;
	    irp->IoStatus.Information = 0;
	    IoCompleteRequest(irp, IO_NO_INCREMENT);
	    return STATUS_SUCCESS;
      }

      while (tcount > 0) {
	    unsigned long rc;
	    int wrote;
	    const long f_pos = offset;

	    if ((tcount >= 4) && (f_pos%4 == 0)) {
		  unsigned long val = *(unsigned long*)bytes;

		  rc = HalSetBusDataByOffset(PCIConfiguration,
					     xsp->bus, xsp->slot.u.AsULONG,
					     &val, f_pos, 4);
		  wrote = 4;

	    } else if ((tcount >= 2) && (f_pos%2 == 0)) {
		  unsigned short val = *(unsigned short*)bytes;

		  rc = HalSetBusDataByOffset(PCIConfiguration,
					     xsp->bus, xsp->slot.u.AsULONG,
					     &val, f_pos, 2);
		  wrote = 2;

	    } else {
		  unsigned char val = *bytes;

		  rc = HalSetBusDataByOffset(PCIConfiguration,
					     xsp->bus, xsp->slot.u.AsULONG,
					     &val, f_pos, 1);
		  wrote = 1;
	    }

	    bytes += wrote;
	    offset += wrote;
	    tcount -= wrote;
      }

      irp->IoStatus.Status = STATUS_SUCCESS;
      irp->IoStatus.Information = irpstack->Parameters.Write.Length - tcount;
      IoCompleteRequest(irp, IO_NO_INCREMENT);

      return STATUS_SUCCESS;
}

static NTSTATUS xxioctl(DEVICE_OBJECT*dev, IRP*irp)
{
      IO_STACK_LOCATION *irpstack = IoGetCurrentIrpStackLocation(irp);
      FILE_OBJECT *file = irpstack->FileObject;
      struct Instance *xsp = (struct Instance*)file->FsContext;

      switch (irpstack->Parameters.DeviceIoControl.IoControlCode) {

	  case PCICONF_SELECT:
	    if (irpstack->Parameters.DeviceIoControl.InputBufferLength
		< sizeof xsp->info) {
		  irp->IoStatus.Status = STATUS_BUFFER_TOO_SMALL;

	    } else {
		  memcpy(&xsp->info, irp->AssociatedIrp.SystemBuffer,
			 sizeof xsp->info);
		  match_device(xsp);
		  irp->IoStatus.Status = xsp->valid_flag
			? STATUS_SUCCESS
			: STATUS_NO_SUCH_DEVICE;
		  irp->IoStatus.Information = 0;
	    }
	    break;

	  default: {
		  IO_ERROR_LOG_PACKET*entry;
		  unsigned long len = sizeof (IO_ERROR_LOG_PACKET);

		  entry = IoAllocateErrorLogEntry(dev, len);
		  entry->ErrorCode = PCICONF_BAD_IOCTL;
		  entry->SequenceNumber = 0;
		  entry->MajorFunctionCode = IRP_MJ_DEVICE_CONTROL;
		  entry->RetryCount = 0;
		  entry->FinalStatus = STATUS_INVALID_PARAMETER;
		  entry->DumpDataSize = sizeof(unsigned long);
		  entry->StringOffset = sizeof (IO_ERROR_LOG_PACKET);
		  entry->NumberOfStrings = 0;

		  entry->DumpData[0] = irpstack->Parameters.DeviceIoControl.IoControlCode;
		  IoWriteErrorLogEntry(entry);

		  irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
		  break;
	    }
      }

      IoCompleteRequest(irp, IO_NO_INCREMENT);
      return STATUS_SUCCESS;
}


static void DriverUnload(PDRIVER_OBJECT drv)
{
      UNICODE_STRING dev_link;

      RtlInitUnicodeString(&dev_link, L"\\DosDevices\\PCICONF");
      IoDeleteSymbolicLink(&dev_link);

      IoDeleteDevice(drv->DeviceObject);
}

NTSTATUS DriverEntry(PDRIVER_OBJECT drv, PUNICODE_STRING path)
{
      struct Instance *xsp;
      UNICODE_STRING dev_name;
      UNICODE_STRING dev_link;
      DEVICE_OBJECT *dev;
      NTSTATUS rc;

      drv->DriverUnload = DriverUnload;
      drv->MajorFunction[IRP_MJ_CREATE] = xxcreate;
      drv->MajorFunction[IRP_MJ_CLOSE]  = xxclose;
      drv->MajorFunction[IRP_MJ_READ]   = xxread;
      drv->MajorFunction[IRP_MJ_WRITE]  = xxwrite;
      drv->MajorFunction[IRP_MJ_DEVICE_CONTROL] = xxioctl;

      RtlInitUnicodeString(&dev_name, L"\\Device\\pciconf");
      RtlInitUnicodeString(&dev_link, L"\\DosDevices\\PCICONF");

      rc = IoCreateDevice(drv, 0, &dev_name, FILE_DEVICE_PCICONF, 0,
			  FALSE, &dev);
      if (rc != STATUS_SUCCESS) return rc;

      dev->Flags = DO_BUFFERED_IO;

      rc = IoCreateSymbolicLink(&dev_link, &dev_name);
      if (rc != STATUS_SUCCESS) {
	    IO_ERROR_LOG_PACKET*entry;
	    unsigned long len = sizeof *entry;

	    entry = IoAllocateErrorLogEntry(dev, len);
	    entry->ErrorCode = PCICONF_NO_LINK;
	    entry->FinalStatus = rc;
	    entry->NumberOfStrings = 0;
	    IoWriteErrorLogEntry(entry);

	    IoDeleteDevice(dev);
	    return rc;
      }

      return STATUS_SUCCESS;
}
