/*
 * grape.c : a simple N-body program to be used with the GRAPE-1
 *           Pipelined N-body engine
 *
 * J. Makino   89/09/16   Ver. 1.00 (created)
 *                        Workbench version with exact force calculation.
 *             89/09/26   Ver. 1.01 
 *                        Running with GRAPE-1 hardware (since 9/22?)
 *                        Optional input for cold collapes calculation
 *	       89/10/02   Ver. 1.02
 *			  Modified for UNIX system, add CPP flags
 *                        EXACT_CALCULATION : not use grape hardware
 *             89/10/12   Ver. 1.03
 *                        Initial power law density supported
 *             90/02/18   Ver. 1.04
 *                        Outer plummer potential field added
 *             90/11/13   Ver. 1.10
 *                        GRAPE-1A interface added (-DGRAPE1A)
 *                        NEMO filestruct interface added (-DNEMOOUT)
 *             00/09/21   Ver. 2.00
 *                        Cleanup of pre-GRAPE-4 stuff
 * 
 *
 */

#ifdef HARP3
#define HARP3 1
#else
#define HARP3 0
#endif

#ifdef GRAPE6
#define GRAPE6 1
#else
#define GRAPE6 0
#endif

#ifndef REAL
#   define REAL double
#endif

#define NMAX 400000

#define NDIM 3
#include <math.h>
double pow();
double sqrt();


#include "../g6hib/grape6.h"

#ifdef NEMO

#include <stdinc.h>
#include <getparam.h>
#include <vectmath.h>
#include <filestruct.h>

/**************** COMMAND LINE PARAMETERS **********************/

char *defv[] = {                /* DEFAULT INPUT PARAMETERS */
    "in=???\n Input file (an atos format ascii snapshot)",
    "out=~\n  Output file(nemo snapshot format, default=no output)",
    "dt=0.03125\n  timestep",
    "tstop=1\n	   time to stop integration",
    "dtout=0.125\n output interval",
    "dtsnapout=0.25\n snapshot output interval",
    "eps2=0.03125\n softening parmeter squared",
    "VERSION=2.0",		/* ID */
    NULL
};
#else

#include <stdio.h>
#endif

#include "grape.h"
static double jerk[NMAX][3];

void print_accel(int n, double acc[][3], double p[])
{
    int i, k;
    puts("calcualted gravity");
    for(i=0; i<n; i++){
	for(k=0; k<3; k++){
	    printf(" %15.8e", acc[i][k]);
	}
	printf(" %15.8e", p[i]);
	printf("\n");
    }
}

static REAL tsys;
static REAL potchange = 0.0;
void print_cpusec()
{
	double cpu;
	second(&cpu);
	
	printf("CPU sec.= %11.2f\n", cpu);
}

void print_cpusec_stderr()
{
	double cpu;
	second(&cpu);
	fprintf(stderr, "CPU sec.= %11.2f\n", cpu);
}

    
void accel_accurate(
    int  n,		/* input	number of particles */
    REAL x[][NDIM],	/* input	position array*/
    REAL m[],		/* input	particle mass */
    REAL eps2,		/* input	softening parameter */
    REAL a[][NDIM])	/* output	calculated acceleration */
{
    int i, j, k;
    REAL dx, dy, dz, r2, r3inv, fi, fj;

/*    puts("(accel_accurate) entered");*/
    for(i=0; i<n; i++){
    	for(k=0;k<NDIM;k++){
    	    a[i][k] = 0.0;
    	}
    }
    for(i=0; i<n-1; i++){
    	for(j=i+1; j<n; j++){
    	    dx = x[i][0] - x[j][0];
    	    dy = x[i][1] - x[j][1];
    	    dz = x[i][2] - x[j][2];
    	    r2 = dx*dx + dy*dy + dz*dz + eps2;
    	    r3inv = 1.0/(sqrt(r2)*r2);
    	    fi = -m[j]*r3inv;
    	    fj =  m[i]*r3inv;
    	    a[i][0] += fi*dx;
    	    a[i][1] += fi*dy;
    	    a[i][2] += fi*dz;
    	    a[j][0] += fj*dx;
    	    a[j][1] += fj*dy;
    	    a[j][2] += fj*dz;
    	}
    }
}

void push_velocity_half(int  n,		/* input	number of particles */
			register REAL v[][NDIM],/* in/out	velocity array */
			register REAL a[][NDIM],/* input	acceleration array */
			REAL dt)/* input	timestep */
{
    register int i;
    register REAL dthalf;
    
    dthalf = dt*0.5;
    for(i=0; i<n; i++){
	v[i][0] += dthalf*a[i][0];
	v[i][1] += dthalf*a[i][1];
	v[i][2] += dthalf*a[i][2];
    }
}


void push_position_full(int  n,	/* input	number of particles */
			REAL x[][NDIM],	/* in/out	position array */
			REAL v[][NDIM],	/* input	velocity array */
			register REAL dt)/* input	timestep */
{
    register int i;

    for(i=0; i<n; i++){
	x[i][0] += dt*v[i][0];
	x[i][1] += dt*v[i][1];
	x[i][2] += dt*v[i][2];
    }
}

void guestimate_acc_etc( int n,
			 double eps2,
			 double m[],
			 double a[][NDIM],
			 double j[][NDIM],
			 double p[])
     /* this routine is for GRAPE-6. Gives some order-estimated values
	force etc */
{
    static int first_call = 1;
    if (first_call){
	int i,k;
	double fscale = m[0]/eps2*sqrt(n+0.0);
	double pscale = m[0]/sqrt(eps2)*sqrt(n+0.0);
	if (fscale < 10.0) fscale = 10.0;
	if (pscale < 10.0) pscale = 10.0;
	first_call = 0;
	for(i=0;i<n;i++){
	    p[i] = pscale;
	    for(k=0;k<NDIM;k++){
		a[i][k] = fscale;
		j[i][k] = fscale;
	    }
	}
    }
}

void calculate_force(
    int  n,		/* input	number of particles */
    REAL x[][NDIM],	/* in/out	position array */
    REAL v[][NDIM],	/* in/out	velocity array */
    REAL a[][NDIM],	/* in/out	acceleration array */
    REAL pot[],		/* in/out	potential array */
    REAL m[],		/* input	mass array */
    REAL eps2)		/* input	softening paramater squared */
{
#ifdef EXACT_CALCULATION
    accel_accurate(n,x,m,eps2,a);
#endif
#if (HARP3)
    h3open_();
    calculate_accel_by_harp3_separate_noopen(n,x,n,x,m,a, pot,eps2);
    h3close_();
#endif    
#if (GRAPE6)
    {
	int clusterid = G6_CLUSTERID;
	guestimate_acc_etc(n,eps2,m,a,jerk,pot);
	g6_open_(&clusterid);
	calculate_accel_by_grape6_noopen(clusterid,n,x,v,m,a,jerk, pot,eps2);
	g6_close_(&clusterid);
    }
    fprintf(stderr,"calc_force , x, p (1) = %e %e %e %e\n",
	    a[0][0],a[0][1],a[0][2],pot[0]);
#endif    
}

void push_system_full(
    int  n,		/* input	number of particles */
    REAL x[][NDIM],	/* in/out	position array */
    REAL v[][NDIM],	/* in/out	velocity array */
    REAL a[][NDIM],	/* in/out	acceleration array */
    REAL pot[],		/* in/out	potential array */
    REAL m[],		/* input	mass array */
    REAL eps2,		/* input	softening paramater squared */
    REAL dt)		/* input	timestep */
{
    REAL outer_potential();
    push_velocity_half(n,v,a,dt);
    push_position_full(n,x,v,dt);
    calculate_force(n,x,v,a,pot,m,eps2);
    push_velocity_half(n,v,a,dt);
}




void obtain_weighted_average(
    int  n,		/* input	number of particles */
    REAL x[][NDIM],	/* input	position array */
    REAL m[],		/* input	mass array */
    double xave[NDIM])	/* output	weighted average */
{
    double msum;
    int i, k;
    for(k=0; k<NDIM; k++)xave[k] = 0.0;
    msum = 0.0;
    for(i=0; i<n; i++){
    	msum += m[i];
    	for(k=0; k<NDIM; k++){
    	    xave[k] += x[i][k]*m[i];
    	}
    }    
    for(k=0; k<NDIM; k++)xave[k] = xave[k]/msum;
}


void diag(REAL time,		/* input system time */
	  int  n,		/* input number of particles */
	  REAL x[][NDIM],	/* input position array */
	  REAL v[][NDIM],	/* input velocity array */
	  REAL m[],		/* input mass array */
	  REAL pot[],		/* input potential array (GRAPE1A only) */
	  REAL eps2,		/* input softening parameter squared */
	  int  mode)		/* input energy output control
				   0 : initial (store energy)
				   1 : print DE/E */
{
    int i,k, k1, k2;
    double cm[NDIM], cmv[NDIM], am[NDIM];
    REAL pe, ke, etot, de;
    REAL pei;
    static REAL etot0;
    REAL outer_potential();
/*    puts("(diag) entered");*/
    printf("Enter diag:"); print_cpusec();
    print_cpusec_stderr();
    /* calculate total potential energy first */
    pe = 0.0;
    for(i=0; i<n; i++){
	pei = pot[i]*0.5;
    	pe += pei*m[i];
    }
    /* calculate totoal kinetic energy */
    /* also add outer potential if specified */
    ke = 0.0;
    for(i=0; i<n; i++){
    	ke+= m[i]*(v[i][0]*v[i][0] + v[i][1]*v[i][1] + v[i][2]*v[i][2]);
    }
    ke *= 0.5;
    etot = pe + ke;
    if(mode == 0){
    	etot0 = etot;
    	de = 0.0;
    }else{
    	de = (etot - etot0 - potchange)/etot;
    }
    printf("T=%12.3f  E = %15.9f  \nDE=%15.8e  V.R. = %10.6f\n", 
            time, etot, de, -ke/pe);
    fprintf(stderr,"T=%12.3f  E = %15.9f  \nDE=%15.8e  V.R. = %10.6f\n", 
            time, etot, de, -ke/pe);
    /* print center of mass etc. */
    obtain_weighted_average(n, x, m, cm);
    obtain_weighted_average(n, v, m, cmv);
    for(k=0; k<NDIM; k++){
    	am[k] = 0.0;
    	k1 = (k+1) % NDIM;
    	k2 = (k+2) % NDIM;
    	for(i=0; i<n; i++){
    	   am[k] += m[i]*(x[i][k1]*v[i][k2] - x[i][k2]*v[i][k1]);
        }
    }
    printf("CM :%12.5e %12.5e %12.5e\n",cm[0],cm[1],cm[2]);
    printf("CMV:%12.5e %12.5e %12.5e\n",cmv[0],cmv[1],cmv[2]);
    printf("AM :%12.5e %12.5e %12.5e\n",am[0],am[1],am[2]);
    printf("Exit  diag:"); print_cpusec();    
    fflush(stdout);
}

#ifdef NEMO

void input_parameters(REAL * dt,		/* output	timestep */
		      REAL * tstop,	/* output	time to stop integration */
		      REAL * dtout,	/* output	output interval */
		      REAL * dtsnapout,	/* output	snapshot output interval */
		      REAL * eps2)	/* output	softening parmeter squared */
{

    REAL eps;
    *dt = getdparam("dt");
    *tstop = getdparam("tstop");
    *dtout = getdparam("dtout");
    *dtsnapout = getdparam("dtsnapout");
    eps = getdparam("eps");
    *eps2 = eps*eps;
    printf("dt=%7.4f tstop=%8.3f dtout=%7.3f dtsnapout=%7.3f\n\
eps=%8.5f \n",
           *dt, *tstop, *dtout, *dtsnapout, eps);
}

#else
void input_parameters(REAL * dt,		/* output	timestep */
		      REAL * tstop,	/* output	time to stop integration */
		      REAL * dtout,	/* output	output interval */
		      REAL * dtsnapout,	/* output	snapshot output interval */
		      REAL * eps2)	/* output	softening parmeter squared */
{

    fprintf(stderr,"Enter dt, tstop, dtout, dtsnapout, eps:");
    scanf("%lf%lf%lf%lf%lf",dt,tstop,dtout,dtsnapout,eps2);
    printf("dt=%7.4f tstop=%8.3f dtout=%7.3f dtsnapout=%7.3f\n\
eps=%8.5f \n",
           *dt, *tstop, *dtout, *dtsnapout, *eps2);
    *eps2 = (*eps2)*(*eps2);
}
#endif

static REAL x[NMAX][NDIM];
static REAL v[NMAX][NDIM];
static REAL m[NMAX];
static REAL a[NMAX][NDIM];
static REAL pot[NMAX];


int nemo_main()
{
    static int  n;
    int pot_out_flag = 0;
    static REAL dt, tstop, dtout, dtsnapout, eps2;
    static REAL time, tnext, tsnapnext;

    timer_init();

    pot_out_flag = 1;

    input_parameters(&dt, &tstop, &dtout,
		     &dtsnapout, &eps2);
    if(set_input_file()){
	puts("snap file open error");
	exit(1);
    }
    if(read_snap(&n,&time,m,x,v,NMAX)){
	puts("snap file read error");
	exit(1);
    }
    if(set_output_snap_file()){
    	puts("snap file open error");
    	exit(1);
    }
    tnext = time + dtout - 0.5*dt;
    tsnapnext = time + dtsnapout - 0.5 * dt;
    calculate_force(n,x,v,a,pot,m,eps2);
    diag(time,n,x,v,m,pot,eps2, 0);
    out_snap_binary(n,time,m,x,v,pot,pot_out_flag);
    while(time < tstop){
        push_system_full(n,x,v,a,pot,m,eps2, dt);
        time += dt;
	tsys = time;
        if(time >= tsnapnext){
            tsnapnext += dtsnapout;
	    out_snap_binary(n,time,m,x,v,pot,pot_out_flag);
        }
        if(time >= tnext){
            tnext += dtout;
            diag(time,n,x,v,m,pot,eps2, 1);
        }
    }
#if HARP3
    h3close_();
#   endif
    return 0;
}

#ifndef NEMO
int main()
{
    return nemo_main();
}
#endif

	
