/*
 * PMULT.C
 *
 * unsigned multiplier unit for predictor pipeline
 *
 * function pmult
 *
 * Copyright Jun Makino 1997
 *
 * Version 1.0 Nov 3 1997
 * Version 1.1 March 1 1998
 * inputs to pmult trancated to physical size of multipliers
 *
 *
 */
#include "grape6sim.h"

/* pmult_with_sign
   fixed point multiplier for predictor pipeline
   driver for pmult. Since pmult does not handle sign bit,
   this function copies the sign of coef toproduct */

ULONG pmult_with_sign(ULONG * product, /* resulted delta t in outbits unsigned integer */
		      ULONG dt, /* dt in DTNBITS integer format, max = 1000... */
		      ULONG coef, /* polynomical coef in inbits unsigned int */
		      ULONG inbits, /* word length for input coef */
		      ULONG outbits /* word length for output */)
{
    ULONG s1, st, pm, err;
    st = 0;
    s1 = (coef >>inbits) & 1; /* sign bit is here */
    err = pmult(&pm, dt, coef & ((1<<inbits)-1), inbits, outbits);
    *product = ((s1 ^ st) << outbits) | pm; /* combine sign bit and product */
    return err;
}

/* pmult
   multiplier for predictor pipeline
   do not handle sign
   perform force-1 rounding */

ULONG pmult(ULONG * product, /* resulted delta t in outbits unsigned integer */
	    ULONG dt, /* dt in DTNBITS integer format, max = 1000... */
	    ULONG coef, /* polynomical coef in inbits unsigned int */
	    ULONG inbits, /* word length for input coef */
	    ULONG outbits /* word length for output */)
{
    ULONG forcebit, mask, prod, nshifts;
    ULONG err = 0;
    /* test if inbits and outbits are acceptable */
    if ((inbits > DTNBITS) ||(outbits > DTNBITS)){
	fprintf(stderr,"pmult: inbits or outbits too large, %d %d\n",
		inbits, outbits);
	err =  1;
    }
    
    /* test if dt is in range */
    if (dt > 1 <<(DTNBITS-1)){
	fprintf(stderr,"pmult: dt too large, 0x%x, %d", dt, DTNBITS);
	err =  2;
	dt &= (ULONG_ONE<<(DTNBITS))-1;
	fprintf(stderr," truncated to, 0x%x\n", dt);
    }
    /* test if coef is in range */
    if (coef >= 1 <<(inbits)){
	fprintf(stderr,"pmult: coef too large, 0x%x, %d\n", coef, inbits);
	coef &= (1 <<(inbits))-1;
	fprintf(stderr," truncated to, 0x%x\n", coef);
	err = 3;
    }
    /* first, we have to round dt. Here we do so-called corrected
       force-1 rounding  */
    dt = force_1_round_and_shift(dt, DTNBITS, inbits);
    dprintf(3,"rounded dt = %x\n", dt);
    /* then, we multiply dt and coef. Note that the length of the
       result is at the maximum inbits + inbits - 1,
       since dt is originally not larger than 1 << (DTNBITS-1)
       */
    prod = dt * coef;
    *product =  force_1_round_and_shift(prod, inbits*2-1, outbits);
    return 0;
}

ULONG pmult_with_sign_by_constant(ULONG * product, /* result  */
				  ULONG coef, /* polynomical coef in inbits s+m int */
				  ULONG mode, /* 0: 2x
						 1: 1.5x
						 2: 0.75x */
				  ULONG inbits, /* word length for input coef */
				  ULONG outbits /* word length for output */)
{

    ULONG s1, st, pm, err;
    st = 0;
    s1 = (coef >>inbits) & 1;
    err = pmult_by_constant(&pm,  coef & ((1<<inbits)-1),
			    mode,  inbits, outbits); 
    *product = ((s1 ^ st) << outbits) | pm;
    return err;
}

ULONG pmult_by_constant(ULONG * product, /* result in outbits unsigned integer */
			ULONG coef, /* polynomical coef in inbits unsigned int */
			ULONG mode, /* 0: 2x
				       1: 1.5x
				       2: 0.75x */
			ULONG inbits, /* word length for input coef */
			ULONG outbits /* word length for output */)
    {
	ULONG forcebit, mask, prod, nshifts;
    /* test if inbits and outbits are acceptable */
    if ((inbits > DTNBITS) ||(outbits > DTNBITS)){
	fprintf(stderr,"pmult: inbits or outbits too large, %d %d\n",
		inbits, outbits);
	return 1;
    }
	
    /* test if coef is in range */
    if (coef >= 1 <<(inbits)){
	fprintf(stderr,"pmult: coef too large, 0x%x, %d\n", coef, inbits);
	return 3;
    }
    if (mode == 0){
      *product = coef *2;
    }else if ((mode ==1) ||(mode ==2)){
      ULONG result_format;
      prod = coef*3;
      if(mode == 1){ /* 1.5x*/
	result_format = inbits + 1;
      }else{
	result_format = inbits + 2;
      }
      *product =  force_1_round_and_shift(prod, result_format, inbits);
    }
    *product &= ((ULONG_ONE <<outbits)-1);
    return 0;
}
    
#ifdef TEST
main()
{
  double fmod();
  ULONG  dt, err, coef, inb, outb, product, mode;
  printf("enter  coef (in hex), inb, outb, mode: ");
  scanf("%lx%ld%ld%ld",&coef, &inb, &outb, &mode);
  err = pmult_by_constant(&product,  coef, mode,inb, outb);
  printf("coef, mode, inb, outb, product  = 0x%lx %ld %ld %ld 0x%lx\n",
	 coef, mode, inb, outb, product);
}
#endif


#ifdef TEST0
main()
{
  double fmod();
  ULONG  dt, err, coef, inb, outb, product;
  printf("enter dt, coef (in hex), inb, outb: ");
  scanf("%lx%lx%ld%ld",&dt,&coef, &inb, &outb);
  err = pmult_with_sign(&product, dt, coef, inb, outb);
  printf("dt,coef, inb, outb, product  = 0x%lx 0x%lx %ld %ld 0x%lx\n",
	 dt, coef, inb, outb, product);
}
#endif


