/*
 * outer_field.c
 *
 * give outer potential field to an N-body system
 *
 */
#include <stdio.h>
#define SHORTREAL
#ifdef SHORTREAL
#define real float
#else
#define real double
#endif

double sqrt();
real plummer_potential(x,x0,m,r0)
    real x[3];
    real x0[3];
    real m;
    real r0;
{
    real r2, dx;
    register int k;
    r2 = r0*r0;
    for(k=0;k<3;k++){
	dx = x[k]-x0[k];
	r2 += dx*dx;
    }
    return (-m/sqrt(r2));
}

real kuzmin_potential(x,x0,m,r0)
    real x[3];
    real x0[3];
    real m;
    real r0;
{
    real r2, dx,dz;
    double fabs();
    register int k;
    dz = fabs(x[2]-x0[2]) + r0;
    r2 = dz*dz;
    for(k=0;k<2;k++){
	dx = x[k]-x0[k];
	r2 += dx*dx;
    }
    return (-m/sqrt(r2));
}

void kuzmin_force(f,x,x0,m,r0)
    real f[3];
    real x[3];
    real x0[3];
    real m;
    real r0;
{
    real r2, dx[3], rfact,dz;
    register int k;
    dz = fabs(x[2]-x0[2]) + r0;
    r2 = dz*dz;
    if(x[2]>x0[2]){
	dx[2]=dz;
    }else{
	dx[2]= -dz;
    }
    for(k=0;k<2;k++){
	dx[k] = x[k]-x0[k];
	r2 += dx[k]*dx[k];
    }
    rfact = -m/(sqrt(r2)*r2);
    for(k=0;k<3;k++){
	f[k] = dx[k]*rfact;
    }
}

void plummer_force(f,x,x0,m,r0)
    real f[3];
    real x[3];
    real x0[3];
    real m;
    real r0;
{
    real r2, dx[3], rfact;
    register int k;
    r2 = r0*r0;
    for(k=0;k<3;k++){
	dx[k] = x[k]-x0[k];
	r2 += dx[k]*dx[k];
    }
    rfact = -m/(sqrt(r2)*r2);
    for(k=0;k<3;k++){
	f[k] = dx[k]*rfact;
    }
}


static real plummer_r0, plummer_m, plummer_center[3];
void set_plummer_field(instr)
    FILE * instr;
{
    fprintf(stderr,"Enter plummer potential parameters:");
#ifdef SHORTREAL
    fscanf(instr,"%f%f%f%f%f",&plummer_r0,&plummer_m,plummer_center,
	   plummer_center+1,plummer_center+2);
#else    
    fscanf(instr,"%lf%lf%lf%lf%lf",&plummer_r0,&plummer_m,plummer_center,
	   plummer_center+1,plummer_center+2);
#endif
    printf("Plummer field parameters, r0=%f, M=%f, center= %f %f %f\n",
	   plummer_r0, plummer_m, plummer_center[0],
	   plummer_center[1],plummer_center[2]);
}

static real tscale;
void set_scale_time(instr)
    FILE * instr;
{
    fprintf(stderr,"Enter potential grow timescale:");
#ifdef SHORTREAL
    fscanf(instr,"%f",&tscale);
#else    
    fscanf(instr,"%lf",&tscale);
#endif
    printf("Potential grow timescale = %g\n", tscale);
    if(tscale > 0.0)     tscale = 1.0/tscale;
}

real scale_function(t)
    real t;
{
    real x, scale;
    real cos();
    x = t*tscale;
    if(x > 1.0){
	scale = 1;
    }else{
	scale = 0.5*(1.0 - cos(x*3.1415926535));
    }
/*    printf("scale res args %g %g %g\n", scale, t, tscale);*/
    return scale;
}
#ifndef PLUMMER
#ifndef KUZMIN
#define PLUMMER
#endif
#endif
#ifdef PLUMMER
#define POTENTIAL_FUNC     plummer_potential
#define FORCE_FUNC     plummer_force
#endif
#ifdef KUZMIN
#define POTENTIAL_FUNC     kuzmin_potential
#define FORCE_FUNC     kuzmin_force
#endif


real outer_potential(x,t)
    real x[3];
    real t;
{
    real pot;
    pot =  POTENTIAL_FUNC(x,plummer_center,plummer_m,plummer_r0);
    if(tscale > 0){
	pot *= scale_function(t);
    }
    return pot;

}
void outer_force(f,x,t)
    real f[3];
    real x[3];
    real t;  
{
    real scale;
    int k;
    FORCE_FUNC(f,x,plummer_center,plummer_m,plummer_r0);
    if(tscale > 0.0){
	scale = scale_function(t);
	for(k=0;k<3;k++) f[k] *= scale;
    }
}
