/*
 * g6control.c
 *
 * test pattern generator for G6 control unit
 *
 * Version 0.00
 * J. Makino 1998 may 5
 *
 * Version 0.1 J. Makino 1998 August 3
 * init_g6_control now sets outout at G6SIM_PIPE_OUT_FILE
 * if G6CONTROL_OUT_FILE is not set
 * Version 0.2 J. Makino 1998 August 10
 * test sequence for WD/VD/ND on FO port
 */

#include "grape6sim.h"
#include "g6chip.h"
#include "ecc64.h"
#include "g6control.h"

FILE * ftestout;

void init_g6control()
{
    static int pout_open = 0;
    set_parity_matrix();
    if(pout_open == 0){
	if (getenv("G6CONTROL_OUT_FILE") != NULL){
	    set_output_file(&ftestout, "G6CONTROL_OUT_FILE");
	}else{
	   set_output_file(&ftestout, "G6SIM_PIPE_OUT_FILE");
	}
	    
	pout_open = 1;
    }
}

void set_jp_physicalid(struct jpw_struct  * jp_unit,
			      int physicalid)
{
    jp_unit->internal_register.physicalid = physicalid;
    fprintf(ftestout,"JPW PHYSICALID set to %x\n", physicalid);
}

void reset_jp_status(struct jpw_struct  * jp_unit)
{
    jp_unit->internal_register.perror = 0;
    jp_unit->internal_register.wcount = 0;
}

void set_jp_control(struct jpw_struct  * jp_unit,
		    int address,
		    int data)
{
    struct jpw_internal_registers * jpc;
    jpc = &(jp_unit->internal_register);
    if (address < 0){
	fprintf(stderr,"set_jp_contrl, impossible address %x\n", address);
	exit(-1);
    }else if  (address <=JPW_MAP_MAX){
	jpc->map[address] = data & 0xf;
    }else if  (address ==JPW_NDATA_ADR){
	jpc->ndata= data & 0x1f;
    }else if  (address ==JPW_VCID_ADR){
	if ((data>>10) == jpc->physicalid){
	    jpc->vcid= data & 0x3ff;
	    fprintf(ftestout,"JPW vcid changed to %d\n", jpc->vcid);
	}
    }else if  (address ==JPW_ADLY_ADR){
	jpc->adly= data & 0x7;
    }else if  (address ==JPW_WDLY_ADR){
	jpc->wdly= data & 0x7;
    
    }else if  (address ==JPW_ODLY_ADR){
	jpc->odly= data & 0xf;
    
    }else if  (address ==JPW_DDLY_ADR){
	jpc->ddly= data & 0x7;
    
    }else{
	fprintf(stderr,"set_jp_contrl, impossible address %x\n", address);
	exit(-1);
    }
    set_jpw_internal_port(&(jp_unit->internal_port), address, data, 1);
    put_jpw_internal_port(&(jp_unit->internal_port));
    set_jpw_internal_port(&(jp_unit->internal_port), address, data, 0);
}

void write_jp_hostport(struct jpw_struct * jp_unit,
		       unsigned int data,
		       unsigned int parity,
		       int check_parity /*if zero, ignore parity  input*/)
{
    struct jpw_internal_registers * jpc;
    jpc = &(jp_unit->internal_register);


    /* first check and set parity error status */
    if(check_parity){
	if (check_byte_parity(data, parity)){
	    jpc->perror |= 1;
	}
    }
    /* output test pattern - add correct paity */
    parity = generate_byte_parity(data);
    set_jpw_host_port(&(jp_unit->host_port), data, parity, 0);
    put_jpw_host_port(&(jp_unit->host_port));
    set_jpw_host_port(&(jp_unit->host_port), data, parity, 1);

    /* then, set data to buffer */
    if (jpc->wcount == 0){
	/* new particle. Clear buffer */
	int i;
	for(i=0;i<JPW_MAXPACKET; i++){
	    jpc->data[i] = 0;
	}
    }
    if (jpc->wcount < 2){
	jpc->data[jpc->wcount] =data;
    }else{
	jpc->data[jpc->map[jpc->wcount-2]+2] = data;
    }
    jpc->wcount ++;
    if (jpc->wcount == JPW_MAXPACKET) jpc->wcount = 0;
    if (jpc->wcount == (jpc->ndata+2)) jpc->wcount = 0;
    if (jpc->wcount == 0){
	dump_jp_memory_port(jp_unit);
    }
}

void dump_jp_memory_port(struct jpw_struct * jp_unit)
{
    int i;
    int imax = JPW_MAXWORDS /2;
    int command_code;
    int memory_address;
    int chipid_mask;
    int chipid;
    int id_matched;
    struct jpw_internal_registers * jpc;
    struct jpw_memory_port * jpm;
    jpc = &(jp_unit->internal_register);
    jpm = &(jp_unit->memory_port);
    
    chipid = jpc->data[0] & 0x3ff;
    chipid_mask = ((jpc->data[0])>>10)&0x3ff;
    command_code = ((jpc->data[0])>>30)&0x3;
    /*    memory_address = (jpc->data[1]<<3) & 0x1fffff;
	  This shift is unnecessary; corrected on 1998/8/22
     */
    memory_address = (jpc->data[1]) & 0x1fffff;

    id_matched = 1;
    if (((jpc->vcid) ^chipid)&chipid_mask){
	/* chipid does not match */
	id_matched = 0;
    }
    
    for(i=0; i <(imax+jpc->odly); i++){
	int adr_index, data_index, we_index;
	jpm->oe = 1;
	adr_index = i - jpc->adly;
	data_index = i - jpc->ddly;
	we_index = i - jpc->wdly;
	if ((adr_index >=0 ) && (adr_index < imax)){
	    jpm->addr = memory_address + adr_index;
	    jpm->addr_valid = 1;
	}else{
	    jpm->addr_valid = 0;
	}
	if ((data_index >=0 ) && (data_index < imax)){
	    /* here I assume that even address is in lower half ... */
	    jpm->data = ((ULONG)(jpc->data[data_index*2+2]))|(((ULONG)(jpc->data[data_index*2+3]))<<32);
	    jpm->parity = generate_parity(jpm->data);
	    jpm->data_valid = 1;
	}else{
	    jpm->data_valid = 0;
	}
	if ((we_index >=0 ) && (we_index < imax)){
	    if (id_matched) jpm->we = 0;
	}else{
	    jpm->we = 1;
	}
	put_jpw_memory_port(jpm);
    }
}

void put_jpw_memory_port(struct jpw_memory_port * jpm)
{
    fprintf(ftestout,"JPMEM ");
    if (jpm->addr_valid) {
	fprintf(ftestout," %6x", jpm->addr & 0x1fffff);
    }else{
	fprintf(ftestout," XXXXXX");
    }
	
    fprintf(ftestout," %1x %1x",  jpm->oe, jpm->we);
    if (jpm->data_valid) {
	fprintf(ftestout," %16lx %2lx", jpm->data, jpm->parity);
    }else{
	fprintf(ftestout," XXXXXXXXXXXXXXXX XX");
    }
    fprintf(ftestout,"\n");
}

void put_jpw_internal_port(struct jpw_internal_port * ji)
{
    fprintf(ftestout,"JPINT ");
    fprintf(ftestout," %2x %8x %1x", ji->ain,ji->din, ji->we );
    fprintf(ftestout,"\n");
}

void set_jpw_internal_port(struct jpw_internal_port * ji,
			 int address,
			 int data,
			 int we)
{
    ji->ain = address;
    ji->din = data;
    ji->we = we;
}

void put_jpw_host_port(struct jpw_host_port * jh)
{
    fprintf(ftestout,"JPHST ");
    fprintf(ftestout," %9lx %1x", jh->data & 0xfffffffffL, jh->we );
    fprintf(ftestout,"\n");
}
    
void set_jpw_host_port(struct jpw_host_port * jh,
		       unsigned int data,
		       unsigned int parity,
		       int we)
{
    jh->data = ((ULONG) data) | (((ULONG) parity)<<32);
    jh->we = we;
}

void set_standard_jp_control(struct jpw_struct * jp_unit)
{
    int i;
    for(i=0;i<JPW_MAXWORDS; i++){
	set_jp_control(jp_unit, i, i );
    }
    set_jp_control(jp_unit, JPW_NDATA_ADR, JPW_MAXWORDS);
    set_jp_control(jp_unit, JPW_ADLY_ADR, ADLY_DEFAULT);
    set_jp_control(jp_unit, JPW_WDLY_ADR, WDLY_DEFAULT);
    set_jp_control(jp_unit, JPW_ODLY_ADR, ODLY_DEFAULT);
    set_jp_control(jp_unit, JPW_DDLY_ADR, DDLY_DEFAULT);
}


initialize_jp_control(struct jpw_struct *  jp_unit)
{
    /* set PHYSID = 1 and VCID = 0 */
    init_g6control();
    set_jp_physicalid(jp_unit, 1);
    reset_jp_status(jp_unit);
    set_standard_jp_control(jp_unit);
    set_jp_control(jp_unit, JPW_VCID_ADR, (1 <<10)|0);
}

void pack_jparticle_to_memory_data(ULONG memdata[8], struct jparticle * jp)
     
{
    int k;
    union float_and_uint fu;
    double mass_double;
    /* position ... */
    for(k=0;k<3;k++){
	memdata[k] = jp->ix[k][0];
    }
    /* v */
    /*
     jp->ix[0][1] = memdata[3]&0xffffffffL;
    jp->ix[1][1] = memdata[3]>>32;*/
    
    memdata[3] = jp->ix[0][1] | (jp->ix[1][1]<<32);
    /*    jp->ix[2][1] = memdata[4]&0xffffffffL;
    jp->ix[0][2] = (memdata[4]>>32)&PRED_A_MASK_U;
    jp->ix[1][2] = (memdata[4]>>53)|((memdata[5]&0x3ffL)<<11);*/
    memdata[4] = (jp->ix[1][2]<<53)|(jp->ix[0][2]<<32)|jp->ix[2][1];
    /*
    jp->ix[2][2] = (memdata[5]>>10) & PRED_A_MASK_U;
        jp->ix[0][3] = (memdata[5]>>31)&PRED_A1_MASK_U;
    jp->ix[1][3] = (memdata[5]>>48)|((memdata[6]&1L)<<16);*/
    memdata[5] = (jp->ix[1][3]<<48)|(jp->ix[0][3]<<31)|(jp->ix[2][2]<<10)|(jp->ix[1][2]>>11);
    /*    
    jp->ix[2][3] = (memdata[6]>>1) & PRED_A1_MASK_U;
    jp->ix[0][4] = (memdata[6]>>18)&PRED_A2_MASK_U;
    jp->ix[1][4] = (memdata[6]>>29)&PRED_A2_MASK_U;
    jp->ix[2][4] = (memdata[6]>>40)&PRED_A2_MASK_U;

    jp->dtjmsb = (memdata[6]>>51)&0x3fL;
    jp->tjlsb = (memdata[6]>>57)&0x1L;*/
    memdata[6] = (jp->tjlsb<<57)|(jp->dtjmsb<<51)|
	(jp->ix[2][4]<<40)|(jp->ix[1][4]<<29)|(jp->ix[0][4]<<18)|
	(jp->ix[2][3]<<1)|(jp->ix[1][3]>>16);
    /*    
    fu.u = (unsigned int) (memdata[7]&0xffffffffL);
    mass_double = fu.f;
    jp->mass = convert_double_to_grape_float(mass_double,INTERACTION_F_LEN_U);
    jp->index = memdata[7]>>32;*/
    fu.f = convert_grape_float_to_double(jp->mass, INTERACTION_F_LEN_U);
    memdata[7] = (jp->index<<32)|((ULONG) fu.u);
}

void construct_jpw_command_header(unsigned int cmdwords[2],
				  int address,
				  struct jpw_struct * jpu)
{
    unsigned int command_code = 0;
    unsigned int chipmask, chipid;
    unsigned int memaddr;
    memaddr = address<<3;
    cmdwords[1] = memaddr;
    chipmask = 0x3ff;
    chipid = jpu->internal_register.vcid;
    cmdwords[0] = (command_code <<30 )|(chipmask<<10) | chipid;
}


int jpcomp(struct jparticle * j1,
	   struct jparticle * j2)
{
    int j, k;
    for(j=0;j<5;j++)
	for(k=0;k<3;k++){
	    if(j1->ix[k][j] != j2->ix[k][j]){
		return 1;
	    }
	}
    if((j1->tjlsb != j2->tjlsb)||
       (j1->dtjmsb != j2->dtjmsb)||
       (j1->index!= j2->index)){
	return 1;
    }
    if(j1->mass != j2->mass){
	ULONG exponent, sign, zero, mantissa;
	decompose_float(j1->mass,INTERACTION_F_LEN_U,&exponent,&sign,&zero,&mantissa);
	if((zero != 1) && (mantissa != 0))return 1;
	decompose_float(j2->mass,INTERACTION_F_LEN_U,&exponent,&sign,&zero,&mantissa);
	if((zero != 1) && (mantissa != 0))return 1;
    }
    return 0;
}
    

void put_jpport_data(int address, struct jparticle * jp, struct chip_control_struct * ccu)
{
    int i;
    struct jpw_struct * jpu;
    ULONG membuf[8];
    unsigned int cmdwords[2];
    struct jparticle jpdum;
    jpu = &(ccu->jpu);
    pack_jparticle_to_memory_data(membuf, jp);
    unpack_jparticle_from_memory_data(&jpdum,membuf);
    if ( jpcomp(jp, &jpdum)){
	fprintf(stderr,"put_jpport_data, translation failed\n");
	printf("original JP\n");
	print_jparticle(jp);
	printf("reconstructed JP\n");
	print_jparticle(&jpdum);
	exit(-1);
    }

    construct_jpw_command_header(cmdwords, address, jpu);
    write_jp_hostport(jpu, cmdwords[0], 0, 0);
    write_jp_hostport(jpu, cmdwords[1], 0, 0);

    for(i=0;i<8; i++){
	write_jp_hostport(jpu, (unsigned int)(membuf[i]&0xffffffffL),0,0);
	write_jp_hostport(jpu, (unsigned int)(membuf[i]>>32),0,0);
	ccu->local_memory[(address<<3)+i] = membuf[i];
    }
}

    
/*
 * part for IP
 */
void reset_ip_status(struct ipw_struct  * ip_unit)
{
    ip_unit->internal_register.perror = 0;
    ip_unit->internal_register.wcount = 0;
}

void write_ip_hostport(struct ipw_struct * ip_unit,
		       unsigned int data,
		       unsigned int parity,
		       int check_parity /*if zero, ignore parity  input*/)
{
    struct ipw_internal_registers * ipc;
    ipc = &(ip_unit->internal_register);


    /* first check and set parity error status */
    if(check_parity){
	if (check_byte_parity(data, parity)){
	    ipc->perror |= 1;
	}
    }
    /* output test pattern - add correct paity */
    parity = generate_byte_parity(data);
    set_ipw_host_port(&(ip_unit->host_port), data, parity, 0);
    put_ipw_host_port(&(ip_unit->host_port));
    set_ipw_host_port(&(ip_unit->host_port), data, parity, 1);

    /* then, set data to buffer */
    if (ipc->wcount == 0){
	ipc->start_address = data;
	ipc->ndata = IPW_MAXPACKET;
    }else  if (ipc->wcount == 1){
	ipc->ndata = data;
    }else{
	ipc->data[ipc->wcount-2] = data;
    }
    ipc->wcount ++;
    if (ipc->wcount == (ipc->ndata+2)) ipc->wcount = 0;
    if (ipc->wcount == 0){
	dump_ip_internal_port(ip_unit);
    }
}

void write_ip_hostport_grape_float(struct ipw_struct * ip_unit,
				   ULONG gfloat,
				   unsigned int parity,
				   int check_parity /*if zero, ignore parity  input*/)
{
    union float_and_uint fu;
    ULONG converted;
    double d;
	
    d = convert_grape_float_to_double(gfloat, INTERACTION_F_LEN_U);
    fu.f = d;
    d = fu.f;
    converted = convert_float_to_grape_float(fu.f, INTERACTION_F_LEN_U);
    if (converted != gfloat){
	fprintf(stderr, "write_ip_hostport_grape_float, conversion failed\n");
	fprintf(stderr, "input = %lx double converted = %lx %lx\n ", gfloat, converted, gfloat ^ converted);
    }
	
    write_ip_hostport(ip_unit, fu.u,  parity, check_parity);
}


dump_ip_internal_port(struct ipw_struct * ip_unit)
{
    int i;
    int command_code;
    int address;
    struct ipw_internal_registers * ipc;
    struct ipw_internal_port * ipi;
    ipc = &(ip_unit->internal_register);
    ipi = &(ip_unit->internal_port);
    address = ipc->start_address;
    for(i=0; i<ipc->ndata; i++){
	ipi->a_valid = ipi->wejp = ipi->wect = ipi->wefo = ipi->wecalc = 0;
	ipi->wetl = ipi->weth = 0;
	if (address < 0x400 ){
	    /* address range for i particle */
	    int address_low;
	    int address_high;
	    int address_translated;
	    int address_low_translated;
	    int iphysp;
	    int k;
	    /* perform address translation */
	    address_low = address & 0xf;
	    address_high = address>>4;
	    address_low_translated = ipc->map[address_low];
	    address_translated = address_high<<4 | address_low_translated;

	    /* set force pipeline address register */
	    ipi->afp = address_translated & 0x7f;

	    if ((address_low_translated < 6) ||(address_low_translated >10)){
		ipi->dout = ipc->data[i];
	    }else{
		double ddata;
		ddata = *((float*)(&(ipc->data[i])));
		ipi->dout = convert_double_to_grape_float(ddata,
							  INTERACTION_F_LEN_U);
	    }

	    /*set write enable */
	    for(k=0;k<NPHYSPIPE; k++){
		ipi->wefp[k] = 0;
	    }
	    iphysp = (address_high>>3) & 7;
	    ipi->wefp[iphysp] = 1;

	    ipi->afp_valid = 1;
	    put_ipw_internal_port(ipi);

	    ipi->afp_valid =  ipi->wefp[iphysp] = 0;

	    /* counting up */
	    address_low ++;
	    if (address_low == ipc->ndata_ip){
		address_low = 0;
		address_high ++;
	    }
	    address = address_low | (address_high<<4);
	}else{
	    /*all except force pipe register write */
	    /* first, decode upper bits */
	    int address_block = address>>10;
	    if (address_block == IPW_IPW_ADDRESS){
		/* IPW internal register write, no external write */
		int address_low = address & 0x3ff;
		if (address_low < IPW_MAXMAP){
		    ipc->map[address_low] = ipc->data[i];
		}else if (address_low == IPW_ND_ADDRESS){
		    ipc->ndata_ip = ipc->data[i];
		}else if (address_low == IPW_TESTMODE_ADDRESS){
		    ipi->testmode = ipc->data[i];
		}
	    }else{
		/* other units, write and return */
		if(address_block == IPW_JPW_ADDRESS){
		    ipi->wejp = 1;
		}else if (address_block == IPW_CTAB_ADDRESS){
		    ipi->wect = 1;
		}else if (address_block == IPW_FO_ADDRESS){
		    ipi->wefo = 1;
		}else if (address_block == IPW_CALC_ADDRESS){
		    ipi->wecalc = 1;
		}else if (address_block == IPW_TI_ADDRESS){
		    if (address & 1){
			ipi->wetl = 1;
		    }else{
			ipi->weth = 1;
		    }
			
		}
		ipi->a = address & 0x3ff;
		ipi->dout = ipc->data[i];
		ipi->a_valid = 1;
		put_ipw_internal_port(ipi);
		ipi->a_valid = ipi->wejp = ipi->wect = ipi->wefo = ipi->wecalc = 0;
		ipi->wetl = ipi->weth = 0;
	    }
	    address ++;
	}
    }
}


void put_ipw_internal_port(struct ipw_internal_port * ipi)
{
    int k;
    fprintf(ftestout,"IPINT ");
    if (ipi->afp_valid){
	fprintf(ftestout," %2x", ipi->afp);
    }else{
	fprintf(ftestout," XX");
    }
    if (ipi->a_valid){
	fprintf(ftestout," %3x", ipi->a);
    }else{
	fprintf(ftestout," XXX");
    }
	


    fprintf(ftestout," %9lx %8x ", ipi->dout,ipi->testmode);
    for(k=0;k<NPHYSPIPE;k++){
	fprintf(ftestout,"%1x",ipi->wefp[k]);
    }
    fprintf(ftestout," %1x%1x%1x%1x%1x%1x",ipi->wejp,ipi->wect,
	    ipi->wefo,ipi->wecalc,ipi->weth,ipi->wetl);
    fprintf(ftestout,"\n");
}

void reset_ipw_internal_port(struct ipw_internal_port * ipi)
{
    int k;
    for(k=0;k<NPHYSPIPE;k++){
	ipi->wefp[k] = 0;
    }
    ipi->wejp = ipi->wect =  ipi->wefo = ipi->wecalc = 0;
    ipi->testmode = 0;
    ipi->a_valid = ipi->afp_valid = 0;
}


void put_ipw_host_port(struct ipw_host_port * ih)
{
    fprintf(ftestout,"IPHST ");
    fprintf(ftestout," %9lx %1x", ih->data & 0xfffffffffL, ih->we );
    fprintf(ftestout,"\n");
}
    
void set_ipw_host_port(struct ipw_host_port * ih,
		       unsigned int data,
		       unsigned int parity,
		       int we)
{
    ih->data = ((ULONG) data) | (((ULONG) parity)<<32);
    ih->we = we;
}

void set_standard_ip_control(struct ipw_struct * ip_unit)
{
    int i;
    write_ip_hostport(ip_unit, IPW_INTERNAL_ADDRESS, 0,0);/* first word: start address */
    write_ip_hostport(ip_unit,IPW_MAXREG, 0,0);/* second word: nwords */
    for(i=0;i<IPW_MAXREG; i++){
	ip_unit->internal_register.map[i] = i;
	write_ip_hostport(ip_unit,i, 0,0);
    }
    ip_unit->internal_register.ndata_ip = IPW_MAXREG;
    write_ip_hostport(ip_unit, IPW_INTERNAL_ADDRESS|IPW_ND_ADDRESS, 0,0);
    write_ip_hostport(ip_unit,1, 0,0);
    write_ip_hostport(ip_unit,IPW_MAXREG, 0,0);
    write_ip_hostport(ip_unit, IPW_INTERNAL_ADDRESS|IPW_TESTMODE_ADDRESS, 0,0);
    write_ip_hostport(ip_unit,1, 0,0);
    write_ip_hostport(ip_unit,0, 0,0);
}

initialize_ip_control(struct ipw_struct * ip_unit)
{
    set_standard_ip_control(ip_unit);
}
void put_ipport_particle(struct gchip * chip,
			 struct chip_control_struct * ccu,
			 int ni,
			 int nvp)
{
    int i,j,k;
    unsigned int data;
    struct ipw_struct * ipu = &(ccu->ipu);
    write_ip_hostport(ipu,0, 0,0);/* first word: start address */
    write_ip_hostport(ipu,IPW_MAXREG*ni, 0,0);/* first word: start address */
    for(i=0;i<ni;i++){
	int ivp = i % nvp;
	int ipp = i/nvp;
	struct iparticle * ip = &(chip->inreg[i]);
	
	for(k=0;k<3;k++){
	    data = ((ULONG)ip->xi[k])>>32;
	    write_ip_hostport(ipu,data, 0,0);
	    data = (ip->xi[k]) &0xffffffffL;
	    write_ip_hostport(ipu,data, 0,0);
	    
	}
	for(k=0;k<3;k++){
	    write_ip_hostport_grape_float(ipu,ip->vi[k], 0,0);
	}
	write_ip_hostport_grape_float(ipu, ip->eps2, 0, 0);
	write_ip_hostport_grape_float(ipu, ip->h2, 0, 0);
	write_ip_hostport(ipu, (unsigned int) ip->index, 0, 0);
	data = ((((ULONG)ip->phiscale) & 0x3ffL)<<20)|
	    ((((ULONG)ip->fscale)&0x3ffL)<<10)|
	    (((ULONG)ip->jscale)&0x3ffL);
	write_ip_hostport(ipu, data, 0, 0);
    }
}

/*
 * part for FO
 */
void set_fo_vcid(struct fo_struct  * fo_unit,
			      int vcid)
{
    fo_unit->internal_register.vcid = vcid;
}

void copy_jpw_vcid_to_fo_vcid(struct chip_control_struct * ccu)
{
    ccu->fou.internal_register.vcid =
	ccu->jpu.internal_register.vcid;
}
    
int get_fo_vcid(struct chip_control_struct * ccu)
{
    return ccu->fou.internal_register.vcid;
}
    
void reset_fo_status(struct fo_struct  * fo_unit)
{
    int i;
    fo_unit->internal_register.inactive = 0;
    for(i=0;i<4;i++){
	fo_unit->internal_register.ests[i] = 0;
    }
    fo_unit->host_port.sts = 1;
}

void set_fo_control(struct fo_struct  * fo_unit,
		    int address,
		    int data)
{
    struct fo_internal_registers * fc;
    fc = &(fo_unit->internal_register);
    if (address < 0){
	fprintf(stderr,"set_jp_contrl, impossible address %x\n", address);
	exit(-1);
    }else if  (address <=FO_MAP_MAX){
	fc->map[address] = data & 0xf;
    }else if  (address ==FO_CMD0_ADR){
	fc->cmd[0]= data ;
    }else if  (address ==FO_CMD1_ADR){
	fc->cmd[1]= data ;
    }else if  (address ==FO_NI_ADR){
	fc->ni= data ;
    }else if  (address ==FO_NW_ADR){
	fc->nw= data ;
    }else if  (address ==FO_INACTIVE_ADR){
	int vcid;
	int mask;
	vcid = (data >> 1) &0x3ff;
	mask = (data >> 11) &0x3ff;
	fprintf(stderr,"set inactive: data=%x, mask=%x vcid1=%x vcid2=%x\n",
		data, mask, vcid, fc->vcid);
	if ( (vcid&mask) == ((fc->vcid)&mask)){
	    fc->inactive= data &1 ; /* need to check VCID here */
	}
	fprintf(stderr,"set inactive=%x,\n", fc->inactive);
	    
    }
    set_fo_internal_port(&(fo_unit->internal_port), address, data, 1);
    put_fo_internal_port(&(fo_unit->internal_port));
    set_fo_internal_port(&(fo_unit->internal_port), address, data, 0);
}

/*
 * fo_port_data: stab routine which always returns
 * address * 16
 */

unsigned int fo_port_data_old(int address)
{
    return address * 16;
}
    
unsigned int fo_port_data(int address,
			  struct gchip * chip)
{
    int local_address, ip;
    unsigned int data;
    ULONG ldata;
    struct pipe_output_register_set * po;
    local_address = address & 15;
    ip = (address >>4)&63;
    po = &(chip->outreg[ip]);
    /* register map:
       0, 1: FXH, FXL
       2,3   FY
       4,5   FZ
       6,7 POTL POTH
       8-10 JX,JY,JZ
       11   RNB
       12   INB
       13   status
       */
    if (local_address < 8){
	if (local_address<6){
	    ldata = po->acc[local_address>>1];
	}else{
	    ldata = po->phi;
	}
	if (local_address &1){
	    data = ldata;
	}else{
	    data = ldata >>32;
	}
    }else if(local_address < 11){
	data = po->jerk[local_address-8];
    }else if(local_address == 11){
	data = (po->rnnb)>>4;
    }else if(local_address == 12){
	data = po->innb;
    }else if(local_address == 13){
	data = ((po->pacc_sum_flag)<<18)|
	    ((po->jacc_sum_flag)<<9)|
	    ((po->facc_sum_flag));
    }else{
	fprintf(stderr,"(fo_port_address) local address %d invalid\n",
		local_address);
	exit(-1);
    }
    return data;
}
    
unsigned int fo_data_with_latency(int address,
				  struct gchip *chip,
				  int latency,
				  int no_initialize,
				  int * is_valid)
{
    static unsigned int delayed_data[FO_MAX_LATENCY];
    static int iaccess = 0;
    int i;
    if (no_initialize == 0){
	iaccess = 0;
    }
    /* shift data */
    for(i=FO_MAX_LATENCY-1;i>0; i--){
	 delayed_data[i] = 	 delayed_data[i-1];
    }
    delayed_data[0] = fo_port_data(address,chip);
    
    if (iaccess < latency) {
	*is_valid = 0;
    }else{
	*is_valid = 1;
    }
    iaccess ++;
    return delayed_data[latency];
}

void dump_fo_pipe_port(struct fo_struct * fo_unit,
		       struct gchip * chip)
{
    int i;
    int imax;
    struct fo_internal_registers * fc;
    struct fo_pipeline_port * fp;
    int address;
    fc = &(fo_unit->internal_register);
    fp = &(fo_unit->pipeline_port);
    imax = fc->nw * fc->ni + FO_PIPE_READ_LATENCY;
    fprintf(ftestout,"dump_fo pipe %x %x %x\n",
	   imax, fc->nw, fc->ni);
    for(address = i = 0;i<imax; i++){
	int address_translated;
	int address_low_translated, address_low, address_high;
	int address_valid;
	int iphysp;
	int k;
	int is_valid;
	unsigned int data;
	/* perform address translation */
	address_low = address & 0xf;
	address_high = address>>4;
	address_low_translated = fc->map[address_low];
	address_translated = address_high<<4 | address_low_translated;
	iphysp = (address_high>>3) & 7;
	data = fo_data_with_latency(address,chip, FO_PIPE_READ_LATENCY,i, &is_valid);
	address_valid = ((imax - i) > FO_PIPE_READ_LATENCY);
	set_fo_pipe_port(fp,address_translated,address_valid,
			 iphysp, data, is_valid);
	put_fo_pipe_port(fp);
	set_fo_pipe_port(fp,address_translated,address_valid,
			 iphysp,  data, 0);
	
	/* counting up */
	address_low ++;
	if (address_low == fc->nw){
	    address_low = 0;
	    address_high ++;
	}
	address = address_low | (address_high<<4);
    }
}



void dump_fo_nbl_port_nowait(struct fo_struct * fo_unit,
		       struct gchip * chip,
		      int mode /* 0: internal, 1: host */)
{
    int i;
    int iunit;
    struct fo_internal_registers * fc;
    struct fo_nbl_port * fp;
    struct fo_host_port * fh;
    int address, addres_valid, is_valid;
    fc = &(fo_unit->internal_register);
    fp = &(fo_unit->nbl_port);
    fh = &(fo_unit->host_port);
    for (iunit = 0; iunit < NNBUNITS; iunit++){
	struct neighbour_memory * nbp;
	int nwords, i;
	int address_valid;
	unsigned int data;
	unsigned int host_data, host_parity;
	int host_active = 0;
	if (fc->inactive)host_active = 1;
	nbp = &(chip->nbmem[iunit]);
	nwords = nbp->nnb;
	/* first put nword information here */

	address_valid = 1;
	is_valid = 1;
	address = FO_NDATA_ADDRESS;
	data = nwords;
	fprintf(ftestout,"dump_fo nbl %d %d %x\n", iunit, nwords, address);
	if (mode == 0){
	    set_fo_nbl_port(fp,address,address_valid,
			    iunit, data, is_valid);
	    put_fo_nbl_port(fp);
	    set_fo_nbl_port(fp,address,address_valid,
			    iunit,  data, 0);
	}else{
	    host_data = (data & 0x3ff) | (iunit<<10)|
		((fo_unit->internal_register.vcid)<<12) |
		(1<<31);
	    host_parity = generate_byte_parity(host_data);
	    set_fo_host_port(fh, host_data,host_parity, 0,0,1,host_active,1);
	    put_fo_host_port(fh);
	}
	
	for(i = 0; i<nwords; i++){
	    address = i*2;
	    data = ((nbp->nbflags[i])&0xffffL) |
		((nbp->overflown & ULONG_ONE)<<16);
	    if (mode == 0){
		set_fo_nbl_port(fp,address,address_valid,
				iunit, data, is_valid);
		put_fo_nbl_port(fp);
	    }else{
		
		host_data = data ;
		host_parity = generate_byte_parity(host_data);
		set_fo_host_port(fh, host_data,host_parity, 0,0,1,host_active,1);
		put_fo_host_port(fh);
	    }
	    address ++;
	    data = nbp->index[i];
	    if (mode == 0){
		set_fo_nbl_port(fp,address,address_valid,
				iunit, data, is_valid);
		put_fo_nbl_port(fp);
		set_fo_nbl_port(fp,address,address_valid,
				iunit, data, 0);
	    }else{
		host_data = data ;
		host_parity = generate_byte_parity(host_data);
		set_fo_host_port(fh, host_data,host_parity, 0,0,1,host_active,1);
		put_fo_host_port(fh);
		set_fo_host_port(fh, host_data,host_parity, 0,1,1,host_active,0);
	    }
	}
    }
}

#define FO_WCMAX 3
#define FO_BMAX  10
#define NNBMAXDATA (NNBMAX*12+20)
void host_fo_port_wait_and_put_data(unsigned int host_data,
				    int host_active,
				    int mode, /*0:init, 1: stack, 2:flush */
				    struct fo_host_port * fh)
{
    static int data[NNBMAXDATA];
    static int dataout[NNBMAXDATA];
    static int vd[NNBMAXDATA];
    static int nd[NNBMAXDATA];
    static int wd[NNBMAXDATA];
    static int ndata;
    static int fo_burst_count = 0;
    int fo_wait_count;
    unsigned int host_parity;
    fo_wait_count = random() % FO_WCMAX;

    
    if(mode <= 1){
	if(mode == 0) ndata = 0;
	data[ndata] = host_data;
	ndata ++;
    }
    if(mode == 2){
	int i, j;
	int ncycles;

	    
	/* initialize handshake data */
	
	for(i=0;i<NNBMAXDATA;i++){
	    dataout[i]=vd[i]=wd[i]=nd[i]=1;
	}
	
	/* first set the wait state... */
	for(i=j=0;i<ndata; i++){
	    int nowait;
	    int nwait=0;
	    int k;
	    nowait = random() %FO_BMAX;
	    if (nowait == 0){
		int k;
		nwait = random() % FO_WCMAX;
		for (k=0;k<nwait;k++){
		    wd[j]=0;
		    j++;
		}
	    }
	    wd[j] = 1;
	    j++;
	}
	ncycles = j+2;
	nd[2] = 0;
	dataout[0]=dataout[1]=data[0];
	for(i=2,j=0;i<ncycles;i++){
	    vd[i]=0;
	    dataout[i] = dataout[i-1];
	    if(wd[i-2] == 1){
		if(i>2)j++;
		if (j<ndata){
		    nd[i] = 0;
		    dataout[i]=data[j];
		}else{
		    vd[i] = 1;
		}
	    }
	}
	for(i=0,j=0;i<ncycles+1;i++){
	    host_parity = generate_byte_parity(dataout[i]);
	    set_fo_host_port(fh, dataout[i],host_parity,
			     vd[i],nd[i], 1,host_active,wd[i]);
	    put_fo_host_port(fh); 
	}
    }
}

void dump_fo_nbl_port(struct fo_struct * fo_unit,
		       struct gchip * chip,
		      int mode /* 0: internal, 1: host */)
{
    int i;
    int iunit;
    int host_port_mode = 0;
    struct fo_internal_registers * fc;
    struct fo_nbl_port * fp;
    struct fo_host_port * fh;
    int address, addres_valid, is_valid;
    unsigned int host_data, host_parity;
    int host_active = 0;
    static int first_call = 1;
    if (first_call){
	first_call = 0;
	srandom(123456789);
    }
    fc = &(fo_unit->internal_register);
    fp = &(fo_unit->nbl_port);
    fh = &(fo_unit->host_port);
    for (iunit = 0; iunit < NNBUNITS; iunit++){
	struct neighbour_memory * nbp;
	int nwords, i;
	int address_valid;
	unsigned int data;
	if (fc->inactive)host_active = 1;
	nbp = &(chip->nbmem[iunit]);
	nwords = nbp->nnb;
	/* first put nword information here */

	address_valid = 1;
	is_valid = 1;
	address = FO_NDATA_ADDRESS;
	data = nwords;
	if (mode == 0){
	    set_fo_nbl_port(fp,address,address_valid,
			    iunit, data, is_valid);
	    put_fo_nbl_port(fp);
	    set_fo_nbl_port(fp,address,address_valid,
			    iunit,  data, 0);
	}else{
	    host_data = (data & 0x3ff) | (iunit<<10)|
		((fo_unit->internal_register.vcid)<<12) |
		(1<<31);
	    host_fo_port_wait_and_put_data(host_data,host_active,host_port_mode,fh);
	    host_port_mode = 1;
	}
	
	for(i = 0; i<nwords; i++){
	    address = i*2;
	    data = ((nbp->nbflags[i])&0xffffL) |
		((nbp->overflown & ULONG_ONE)<<16);
	    if (mode == 0){
		set_fo_nbl_port(fp,address,address_valid,
				iunit, data, is_valid);
		put_fo_nbl_port(fp);
	    }else{
		
		host_data = data ;
		host_fo_port_wait_and_put_data(host_data,host_active,host_port_mode,fh);
	    }
	    address ++;
	    data = nbp->index[i];
	    if (mode == 0){
		set_fo_nbl_port(fp,address,address_valid,
				iunit, data, is_valid);
		put_fo_nbl_port(fp);
		set_fo_nbl_port(fp,address,address_valid,
				iunit, data, 0);
	    }else{
		host_data = data ;
		host_fo_port_wait_and_put_data(host_data,host_active,host_port_mode,fh);
	    }
	}
    }
    if (mode != 0){
	host_fo_port_wait_and_put_data(host_data,host_active,2,fh);
    }

}


	    



void dump_fo_host_port(struct fo_struct * fo_unit,
		       struct gchip * chip)
{
    int i;
    int imax;
    struct fo_internal_registers * fc;
    struct fo_host_port * fh;
    int address;
    fc = &(fo_unit->internal_register);
    fh = &(fo_unit->host_port);
    imax = fc->nw * fc->ni ;
    for(address = i = 0;i<imax; i++){
	int address_translated;
	int address_low_translated, address_low, address_high;
	int k;
	int is_valid;
	unsigned int data;
	unsigned int parity;
	int active=0;
	if (fc->inactive)active = 1;
	/* perform address translation */
	address_low = address & 0xf;
	address_high = address>>4;
	address_low_translated = fc->map[address_low];
	address_translated = address_high<<4 | address_low_translated;
	data = fo_data_with_latency(address, chip, 0,i, &is_valid);
	parity = generate_byte_parity(data);
	set_fo_host_port(fh, data,parity, 0,0,1,active,0);
	put_fo_host_port(fh);
	set_fo_host_port(fh, data,parity, 0,1,1,active,0);
	
	/* counting up */
	address_low ++;
	if (address_low == fc->nw){
	    address_low = 0;
	    address_high ++;
	}
	address = address_low | (address_high<<4);
    }
}

void set_fo_pipe_port(struct fo_pipeline_port * fp,
		      int address,
		      int address_valid,
		      int iphysical,
		      unsigned int data,
		      int is_valid)
{
    int i;
    for(i = 0; i < NPHYSPIPE; i++){
	fp->is_valid[i] = 0;
    }
    fp->a = address & 0x7f;
    fp->address_valid = address_valid;
    fp->din[iphysical] = data;
    fp->is_valid[iphysical] = is_valid;
}

void put_fo_pipe_port(struct fo_pipeline_port * fp)
{
    int i;
    fprintf(ftestout,"FOPIP ");
    if (fp->address_valid){
	fprintf(ftestout,"%2x", fp->a & 0x7f);
    }else{
	fprintf(ftestout,"XX");
    }
	
    for(i=0;i<NPHYSPIPE;i++){
	if (fp->is_valid[i]) {
	    fprintf(ftestout," %8x", fp->din[i]);
	}else{
	    fprintf(ftestout," XXXXXXXX");
	}
    }
    fprintf(ftestout,"\n");
}


void set_fo_nbl_port(struct fo_nbl_port * fp,
		      int address,
		      int address_valid,
		      int inbl,
		      unsigned int data,
		      int is_valid)
{
    int i;
    for(i = 0; i < NNBUNITS; i++){
	fp->is_valid[i] = 0;
    }
    fp->a = address & 0x3ff;
    fp->address_valid = address_valid;
    fp->din[inbl] = data;
    fp->is_valid[inbl] = is_valid;
}

    

void put_fo_nbl_port(struct fo_nbl_port * fp)
{
    int i;
    fprintf(ftestout,"FONBL ");
    if (fp->address_valid){
	fprintf(ftestout,"%3x", (fp->a) & 0xfff);
    }else{
	fprintf(ftestout,"XXX");
    }
	
    for(i=0;i<NNBUNITS;i++){
	if (fp->is_valid[i]) {
	    fprintf(ftestout," %8x", fp->din[i]);
	}else{
	    fprintf(ftestout," XXXXXXXX");
	}
    }
    fprintf(ftestout,"\n");
}

    

void put_fo_internal_port(struct fo_internal_port * fi)
{
    fprintf(ftestout,"FOINT ");
    fprintf(ftestout," %2x %8x %1x", fi->ain,fi->din, fi->we );
    fprintf(ftestout,"\n");
}

void set_fo_internal_port(struct fo_internal_port * fi,
			 int address,
			 int data,
			 int we)
{
    fi->ain = address;
    fi->din = data;
    fi->we = we;
}

void put_fo_host_port(struct fo_host_port * fh)
{
    fprintf(ftestout,"FOHST ");
    if (fh->vd == 0){
	fprintf(ftestout,"%9lx ", fh->data & 0xfffffffffL);
    }else{
	fprintf(ftestout,"XXXXXXXXX ");
    }
    fprintf(ftestout,"%1x%1x%1x%1x%1x", fh->vd, fh->nd, fh->sts,
	    fh->active, fh->wd );
    fprintf(ftestout,"\n");
}
    
void set_fo_host_port(struct fo_host_port * fh,
		      unsigned int data,
		      unsigned int parity,
		      int vd,
		      int nd,
		      int sts,
		      int active,
		      int wd)

{
    fh->data = ((ULONG) data) | (((ULONG) parity)<<32);
    fh->vd = vd;
    fh->nd = nd;
    fh->sts = sts;
    fh->active = active;
    fh->wd = wd;
}



void set_standard_fo_control(struct fo_struct * fo_unit)
{
    int i;
    set_fo_vcid(fo_unit, 0);
    for(i=0;i<FO_MAXWORDS; i++){
	set_fo_control(fo_unit, i, i );
    }
    set_fo_control(fo_unit, FO_NW_ADR, 14);
    set_fo_control(fo_unit, FO_NI_ADR, 1);
    set_fo_control(fo_unit, FO_INACTIVE_ADR, 0);
}


/*
 * part for CALC
 */

void reset_calc_status(struct calc_struct  * calc_unit)
{
    int i;
    calc_unit->internal_register.n= 0;
    calc_unit->internal_register.ests[0] = 0;
    calc_unit->internal_register.ests[1] = 0;
}

void set_calc_control(struct calc_struct  * calc_unit,
		    int address,
		    int data)
{
    struct calc_internal_registers * fc;
    fc = &(calc_unit->internal_register);
    if (address < 0){
	fprintf(stderr,"set_jp_contrl, impossible address %x\n", address);
	exit(-1);
    }else if  (address ==CALC_LRAM_ADR){
	fc->lram= data ;
    }else if  (address ==CALC_LFORCE_ADR){
	fc->lforce= data ;
    }else if  (address ==CALC_N_ADR){
	fc->n= data*8-1 ;
    }
    set_calc_internal_port(&(calc_unit->internal_port), address, data, 1);
    put_calc_internal_port(&(calc_unit->internal_port));
    set_calc_internal_port(&(calc_unit->internal_port), address, data, 0);
}

/*
 * calc_memory_port_data: stab routine which always returns
 * address * 16
 */



void unpack_jparticle_from_memory_data(struct jparticle * jp,
				       ULONG memdata[8])
{
    int k;
    union float_and_uint fu;
    double mass_double;
    for(k=0;k<8;k++){
	fprintf(ftestout,"memdata[%1d]= %16lx\n", k, memdata[k]);
    }
    /* position ... */
    for(k=0;k<3;k++){
	jp->ix[k][0] = memdata[k];
    }
    /* v */
    jp->ix[0][1] = memdata[3]&0xffffffffL;
    jp->ix[1][1] = memdata[3]>>32;
    jp->ix[2][1] = memdata[4]&0xffffffffL;
    /* a */
    jp->ix[0][2] = (memdata[4]>>32)&PRED_A_MASK_U;
    jp->ix[1][2] = (memdata[4]>>53)|((memdata[5]&0x3ffL)<<11);
    jp->ix[2][2] = (memdata[5]>>10) & PRED_A_MASK_U;
    /* a1 */
    jp->ix[0][3] = (memdata[5]>>31)&PRED_A1_MASK_U;
    jp->ix[1][3] = (memdata[5]>>48)|((memdata[6]&1L)<<16);
    jp->ix[2][3] = (memdata[6]>>1) & PRED_A1_MASK_U;
    /* a2 */
    jp->ix[0][4] = (memdata[6]>>18)&PRED_A2_MASK_U;
    jp->ix[1][4] = (memdata[6]>>29)&PRED_A2_MASK_U;
    jp->ix[2][4] = (memdata[6]>>40)&PRED_A2_MASK_U;

    jp->dtjmsb = (memdata[6]>>51)&0x3fL;
    jp->tjlsb = (memdata[6]>>57)&0x1L;
    
    fu.u = (unsigned int) (memdata[7]&0xffffffffL);
    mass_double = fu.f;
    /*    jp->mass = convert_double_to_grape_float(mass_double,INTERACTION_F_LEN_U);*/
    jp->mass = convert_float_to_grape_float(fu.f,INTERACTION_F_LEN_U);
    jp->index = memdata[7]>>32;
    fprintf(ftestout,"reconstructed JP\n");
    print_jparticle(jp);
}


ULONG calc_memory_data(int address,
		       struct chip_control_struct *ccu)
{
    if(ccu == NULL){
	return (ULONG)(address * 16);
    }else{
	return ccu->local_memory[address];
    }
}
    
ULONG calc_memory_data_with_latency(int address,
				    struct chip_control_struct *ccu,
				    int latency,
				    int no_initialize,
				    int * is_valid)
{
    static ULONG delayed_data[CALC_MAX_LATENCY];
    static int iaccess = 0;
    int i;
    if (no_initialize == 0){
	iaccess = 0;
    }
    /* shift data */
    for(i=CALC_MAX_LATENCY-1;i>0; i--){
	 delayed_data[i] = 	 delayed_data[i-1];
    }
    delayed_data[0] = calc_memory_data(address,ccu);
    
    if (iaccess < latency) {
	*is_valid = 0;
    }else{
	*is_valid = 1;
    }
    iaccess ++;
    return delayed_data[latency];
}

void do_calc(struct chip_control_struct * ccu)
{
    int i;
    int iend;
    struct calc_struct *calc_unit;
    struct calc_internal_registers * cc;
    struct calc_memory_port * cm;
    struct calc_predictor_port * cp;
    int address;
    int vd;
    int run = 1;
    struct jparticle jp;
    calc_unit= &(ccu->calcu);
    cc = &(calc_unit->internal_register);
    cp = &(calc_unit->predictor_port);
    cm = &(calc_unit->memory_port);
    iend = cc->n + cc->lram+1;
    for(i = 0; i <iend; i++){
	int memaddr;
	int addr_valid;
	int is_valid;
	int bufindex;
	ULONG dbuf[8];
	ULONG data;
	memaddr = cc->n - i;
	addr_valid = 1;
	if (memaddr < 0) {
	    addr_valid = 0;
	    memaddr = 0;
	}
	data = calc_memory_data_with_latency(memaddr, ccu,cc->lram,i, &is_valid);
	bufindex = (i-(cc->lram))%8;
	dbuf[7-bufindex] = data;
	set_calc_memory_port(cm,memaddr,addr_valid, data, is_valid);
	put_calc_memory_port(cm);
	if (bufindex == 7){
	    int k;
	    unpack_jparticle_from_memory_data(&jp, dbuf);
	    for(k=0;k<3;k++){
		if (k == 0) {
		    vd = 1;
		}else {
		    vd = 0;
		}
		set_calc_predictor_port(cp, &jp,k,  vd,run);
		put_calc_predictor_port(cp);
	    }
	}
    }
    vd = 0;
    run = 0;
    set_calc_predictor_port(cp, &jp,2,  vd,run);
}

void set_calc_memory_port(struct calc_memory_port * cm,
			  int address,
			  int address_valid,
			  ULONG data,
			  int data_valid)
{
    cm->addr =  address;
    cm->data = data;
    cm->parity = generate_parity(data);
    cm->address_valid = address_valid;
    cm->data_valid = data_valid;
}

void put_calc_memory_port(struct calc_memory_port * cm)
{
    int i;
    fprintf(ftestout,"CALCM ");
    if (cm->address_valid){
	fprintf(ftestout,"%6x", cm->addr & 0xffffff);
    }else{
	fprintf(ftestout,"XXXXXX");
    }
	
    if (cm->data_valid	) {
	fprintf(ftestout," %16lx %2lx", cm->data, cm->parity);
    }else{
	fprintf(ftestout," XXXXXXXXXXXXXXXX XX");
    }
    fprintf(ftestout,"\n");
}

    

void put_calc_internal_port(struct calc_internal_port * ci)
{
    fprintf(ftestout,"CALCI ");
    fprintf(ftestout," %2x %8x %1x", ci->ain,ci->din, ci->we );
    fprintf(ftestout,"\n");
}

void set_calc_internal_port(struct calc_internal_port * fi,
			 int address,
			 int data,
			 int we)
{
    fi->ain = address;
    fi->din = data;
    fi->we = we;
}

void put_calc_predictor_port(struct calc_predictor_port * cp)
{
    int i;
    fprintf(ftestout,"CALCP ");
    fprintf(ftestout," %16lx", cp->ix[0]);
    fprintf(ftestout," %8lx", cp->ix[1]);
    fprintf(ftestout," %6lx", cp->ix[2]);
    fprintf(ftestout," %5lx", cp->ix[3]);
    fprintf(ftestout," %4lx", cp->ix[4]);
    fprintf(ftestout," %2lx %9lx %8lx", cp->tj, cp->mass, cp->index);
    fprintf(ftestout," %1x %1x", cp->vd, cp->run);
    fprintf(ftestout,"\n");
}
    
void set_calc_predictor_port(struct calc_predictor_port * cp,
			     struct jparticle * jp,
			     int coord_index,
			     int vd,
			     int run)
{
    int i;
    for (i=0;i<5;i++){
	cp->ix[i] = jp->ix[coord_index][i];
    }
    cp->mass = jp->mass;
    cp->index = jp->index;
    cp->tj = (jp->tjlsb)<<6 | ((jp->dtjmsb)&0x3fL);
    cp-> vd = vd;
    cp-> run = run;
}


void initialize_chip_control_old(struct chip_control_struct *ccu)
{
    initialize_jp_control(&(ccu->jpu));
    initialize_ip_control(&(ccu->ipu));
}


void interpret_ipw_hostport_data(int start_address,
				 int nwords,
				 int data[],
				 struct chip_control_struct *ccu)
{
    int i;
    int address;
    struct ipw_struct * ipu = &(ccu->ipu);
    if (start_address < IPW_INTERNAL_ADDRESS){
	fprintf(stderr,"interpret_ipw_hostport_data, IPARTICLE address %x is not for this routine",
		start_address);
	exit(-1);
    }
    write_ip_hostport(ipu, start_address, 0, 0);/* first word: start address*/
    write_ip_hostport(ipu, nwords, 0, 0);/* second word: nwords */
    for(i=0;i<nwords; i++){
	write_ip_hostport(ipu, data[i], 0, 0);
    }
    for(i=0, address=start_address;i<nwords; i++,address++){
	int top_address = address & IPW_UNIT_ADDRESS_MASK;
	if(top_address == IPW_INTERNAL_ADDRESS){
	    /* IPW internal register already handled in write_ip_hostport... */
	}else if (top_address == IPW_JPW_INTERNAL_ADDRESS){
	    set_jp_control(&(ccu->jpu), address & IPW_LOCAL_ADDRESS_MASK, data[i]);
	}else if (top_address == IPW_CUTOFF_INTERNAL_ADDRESS){
	    /* do nothing here... yet! */
	}else if (top_address == IPW_FO_INTERNAL_ADDRESS){
	    set_fo_control(&(ccu->fou), address & IPW_LOCAL_ADDRESS_MASK, data[i]);
	}else if (top_address == IPW_CALC_INTERNAL_ADDRESS){
	    set_calc_control(&(ccu->calcu), address & IPW_LOCAL_ADDRESS_MASK, data[i]);
	}
    }
}

void initialize_fo_control_through_ipw(struct chip_control_struct * ccu)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    static int map_data[FO_MAXWORDS];
    for(i=0;i<FO_MAXWORDS; i++)map_data[i] = i;
    printf("initialize_fo_control_through_ipw\n");
    set_fo_vcid(&(ccu->fou), ccu->jpu.internal_register.vcid);
    interpret_ipw_hostport_data(IPW_FO_INTERNAL_ADDRESS, FO_MAXWORDS, map_data, ccu);
    map_data[0] = 1;
    map_data[1] = 14;
    map_data[2] = 0;
    interpret_ipw_hostport_data(IPW_FO_INTERNAL_ADDRESS|FO_NI_ADR,3, map_data, ccu);
}

void send_nbread_through_ipw(struct chip_control_struct * ccu,
			     int id, int mask)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    unsigned int command_data[2];
    command_data[0] = (mask << 10) | id;
    command_data[1] = 0;
    interpret_ipw_hostport_data(IPW_FO_INTERNAL_ADDRESS|FO_CMD0_ADR, 2,
				(int*)command_data, ccu);
}
void send_fo_ni_through_ipw(struct chip_control_struct * ccu,int ni)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    interpret_ipw_hostport_data(IPW_FO_INTERNAL_ADDRESS|FO_NI_ADR, 1,&ni, ccu);
}



void initialize_calc_control_through_ipw(struct chip_control_struct * ccu)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    static int map_data[2];
    map_data[0] = CALC_LRAM_DEFAULT;
    map_data[1] = CALC_LFORCE_DEFAULT;
    printf("initialize_calc_control_through_ipw\n");
    interpret_ipw_hostport_data(IPW_CALC_INTERNAL_ADDRESS, 2, map_data, ccu);
}

void write_calc_through_ipw(struct chip_control_struct * ccu,
			    int address,
			    unsigned int data)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    static int map_data[1];
    map_data[0] = data;
    interpret_ipw_hostport_data(IPW_CALC_INTERNAL_ADDRESS+address, 1, map_data, ccu);
}

void write_ti_through_ipw(struct chip_control_struct * ccu,
			    ULONG ti)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    unsigned int map_data[2];
    map_data[0] = ti>>32;
    map_data[1] = ti& 0xffffffffL;
    interpret_ipw_hostport_data(IPW_TI_INTERNAL_ADDRESS, 2, (int*)map_data, ccu);
}

void write_rscale_through_ipw(struct chip_control_struct * ccu,
			    ULONG rscale)
{
    int i;
    struct ipw_struct * ipu = &(ccu->ipu);
    unsigned int map_data[2];
    map_data[0] = rscale;

    interpret_ipw_hostport_data(IPW_RSCALE_INTERNAL_ADDRESS, 1, (int*)map_data, ccu);
}


void dump_cutoff_table_in_host_port(struct chip_control_struct *ccu)
{
    int i, address;
    write_ip_hostport(&(ccu->ipu),IPW_CUTOFF_INTERNAL_ADDRESS , 0, 0);/* first word: start address*/
    write_ip_hostport(&(ccu->ipu), 128, 0, 0);/* second word: nwords */
    for(i=0;i<128; i++){
	write_ip_hostport(&(ccu->ipu), cutoff_data(i), 0, 0);
    }
}
    

void initialize_chip_control(struct chip_control_struct *ccu)
{
#ifndef PHYSICALID    
#define PHYSICALID 0x123
#endif    
    static int jp_init_data[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,
				 16,
				 PHYSICALID<<10,/*vcid*/
				  ADLY_DEFAULT,
				 WDLY_DEFAULT,
				 ODLY_DEFAULT,
				 DDLY_DEFAULT};
    printf("initialize_chip_control, called\n");
    init_g6control();
    set_jp_physicalid(&(ccu->jpu), PHYSICALID);
    reset_jp_status(&(ccu->jpu));
    interpret_ipw_hostport_data(IPW_JPW_INTERNAL_ADDRESS, 22,jp_init_data,ccu);
    printf("call initialize_ip_control\n");
    
    initialize_ip_control(&(ccu->ipu));
    printf("call initialize_fp_control_through\n");
    initialize_fo_control_through_ipw(ccu);
    printf("call initialize_calc_control_through\n");
    initialize_calc_control_through_ipw(ccu);
}

	    
	




    
#ifdef TEST_IPW
void ipw_put_memo(char* string)
{
    fprintf(ftestout,"IPW_MEMO:");
    fprintf(ftestout,string);
    fprintf(ftestout,"\n");
}

main()
{
    int i;
    struct ipw_struct ip_unit;
    init_g6control();
    reset_ip_status(&ip_unit);
    reset_ipw_internal_port(&(ip_unit.internal_port));
    set_standard_ip_control(&ip_unit);

    ipw_put_memo("test of sending two particles");

    write_ip_hostport(&ip_unit, 0, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 26, 0, 0);/* second word: nwords */

    for(i=0;i<26;i++){
	if (((i%13) == 11) || ((i%13) < 6)){
	    write_ip_hostport(&ip_unit, i, 0, 0);
	}else{
	    /* float data to be converted ... */
	    FLOAT_AND_UINT x;
	    x.f = 0;
	    write_ip_hostport(&ip_unit, x.u, 0, 0);
	}
    }
    ipw_put_memo("test of rewriting map");

    write_ip_hostport(&ip_unit, 0x410, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 1, 0, 0);/* second word: nwords */
    write_ip_hostport(&ip_unit, 7, 0, 0);
    write_ip_hostport(&ip_unit, 0x406, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 1, 0, 0);/* second word: nwords */
    write_ip_hostport(&ip_unit, 0xb, 0, 0);

    ipw_put_memo("test of two particles");
    write_ip_hostport(&ip_unit, 0, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 14, 0, 0);/* second word: nwords */
    for(i=0;i<14;i++){
	write_ip_hostport(&ip_unit, i, 0, 0);
    }
    ipw_put_memo("test of two particles to physical pipe 5");
    write_ip_hostport(&ip_unit, 0x280, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 14, 0, 0);/* second word: nwords */
    for(i=0;i<14;i++){
	write_ip_hostport(&ip_unit, i, 0, 0);
    }
    ipw_put_memo("test of sending data to jpw");
    write_ip_hostport(&ip_unit, 0x800, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 10, 0, 0);/* second word: nwords */
    for(i=0;i<10;i++){
	write_ip_hostport(&ip_unit, 1<<i, 0, 0);
    }
    ipw_put_memo("test of sending data to cut off table");
    write_ip_hostport(&ip_unit, 0xc00, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 64, 0, 0);/* second word: nwords */
    for(i=0;i<64;i++){
	write_ip_hostport(&ip_unit, 1<<(i%32), 0, 0);
    }
    ipw_put_memo("test of sending command to FO  unit");
    write_ip_hostport(&ip_unit, 0x1010, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 2, 0, 0);/* second word: nwords */
    write_ip_hostport(&ip_unit, 1, 0, 0);
    write_ip_hostport(&ip_unit, 2, 0, 0);
    ipw_put_memo("test of sending command to CALC  unit");
    write_ip_hostport(&ip_unit, 0x1400, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 2, 0, 0);/* second word: nwords */
    write_ip_hostport(&ip_unit, 1, 0, 0);
    write_ip_hostport(&ip_unit, 2, 0, 0);
    ipw_put_memo("test of write ti and tj  unit");
    write_ip_hostport(&ip_unit, 0x1800, 0, 0);/* first word: start address*/
    write_ip_hostport(&ip_unit, 2, 0, 0);/* second word: nwords */
    write_ip_hostport(&ip_unit, 1, 0, 0);
    write_ip_hostport(&ip_unit, 2, 0, 0);
}
#endif
#ifdef TEST_JPW
void jpw_put_memo(char* string)
{
    fprintf(ftestout,"JPW_MEMO:");
    fprintf(ftestout,string);
    fprintf(ftestout,"\n");
}

#define JPW_PHID 42
main()
{
    int i;
    struct jpw_struct jp_unit;
    init_g6control();
    set_jp_physicalid(&jp_unit, JPW_PHID);
    reset_jp_status(&jp_unit);
    set_standard_jp_control(&jp_unit);
    set_jp_control(&jp_unit, JPW_VCID_ADR, 0);
    jpw_put_memo("Send data for one particle from zero address");
    write_jp_hostport(&jp_unit, 0, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0, 0, 0);/* second command word */
    for(i=0;i<JPW_MAXWORDS; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    jpw_put_memo("Set VCID to 0");
    set_jp_control(&jp_unit, JPW_VCID_ADR, (JPW_PHID <<10)|0);
    jpw_put_memo("Send data for one particle for 0xffff, vcid=0");
    write_jp_hostport(&jp_unit, 0xffc00, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0xffff, 0, 0);/* second command word */
    for(i=0;i<JPW_MAXWORDS; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    jpw_put_memo("Set VCID to 1");
    set_jp_control(&jp_unit, JPW_VCID_ADR, (JPW_PHID <<10)|1);
    jpw_put_memo("Send data for one particle for 0xffff with VCID 0");
    write_jp_hostport(&jp_unit, 0xffc00, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0xffff, 0, 0);/* second command word */
    for(i=0;i<JPW_MAXWORDS; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    
    jpw_put_memo("Set VCID to 99");
    set_jp_control(&jp_unit, JPW_VCID_ADR, (JPW_PHID <<10)|99);
    jpw_put_memo("Send data for one particle for 0xffff with VCID 99");
    write_jp_hostport(&jp_unit, 0xffc00 | 99, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0xffff, 0, 0);/* second command word */
    for(i=0;i<JPW_MAXWORDS; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    
    jpw_put_memo("Redefine ndata  --- send  8 words only");
    set_jp_control(&jp_unit, JPW_NDATA_ADR, 8);
    jpw_put_memo("Send data for one particle for 0xffff with VCID 99");
    write_jp_hostport(&jp_unit, 0xffc00 | 99, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0xffff, 0, 0);/* second command word */
    for(i=0;i<8; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    
    jpw_put_memo("Redefine table  --- send  even addresses only");
    set_jp_control(&jp_unit, 0, 0);
    set_jp_control(&jp_unit, 1, 2);
    set_jp_control(&jp_unit, 2, 4);
    set_jp_control(&jp_unit, 3, 6);
    set_jp_control(&jp_unit, 4, 8);
    set_jp_control(&jp_unit, 5, 10);
    set_jp_control(&jp_unit, 6, 12);
    set_jp_control(&jp_unit, 7, 14);
    jpw_put_memo("Send data for one particle for 0xffff with VCID 99");
    write_jp_hostport(&jp_unit, 0xffc00 | 99, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0xffff, 0, 0);/* second command word */
    for(i=0;i<8; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    
    jpw_put_memo("Redefine table  --- send  odd addresses only");
    set_jp_control(&jp_unit, 0, 1);
    set_jp_control(&jp_unit, 1, 3);
    set_jp_control(&jp_unit, 2, 5);
    set_jp_control(&jp_unit, 3, 7);
    set_jp_control(&jp_unit, 4, 9);
    set_jp_control(&jp_unit, 5, 11);
    set_jp_control(&jp_unit, 6, 13);
    set_jp_control(&jp_unit, 7, 15);
    jpw_put_memo("Send data for one particle for 0xffff with VCID 99");
    write_jp_hostport(&jp_unit, 0xffc00 | 99, 0, 0);/* first command word */
    write_jp_hostport(&jp_unit, 0xffff, 0, 0);/* second command word */
    for(i=0;i<8; i++){
	write_jp_hostport(&jp_unit, i, 0, 0);
    }
    
}
#endif
#ifdef TEST_FO
void fo_put_memo(char* string)
{
    fprintf(ftestout,"FO_MEMO:");
    fprintf(ftestout,string);
    fprintf(ftestout,"\n");
}

#define FO_PHID 42
main()
{
    int i;
    struct fo_struct fo_unit;
    struct gchip chip;
    init_g6control();
    reset_fo_status(&fo_unit);
    fo_put_memo("For one particle ");
    set_standard_fo_control(&fo_unit);
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);
    fo_put_memo("For 16 particles ");
    set_fo_control(&fo_unit, FO_NI_ADR, 16);
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);
    fo_put_memo("Read force & potential, skip all else, NI=2 ");
    set_fo_control(&fo_unit, FO_NI_ADR, 2);
    set_fo_control(&fo_unit, FO_NW_ADR, 8);
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);
    fo_put_memo("Skip jerk, NI=2 ");
    set_fo_control(&fo_unit, FO_NI_ADR, 2);
    set_fo_control(&fo_unit, FO_NW_ADR, 11);
    set_fo_control(&fo_unit, 8, 11);
    set_fo_control(&fo_unit, 9, 12);
    set_fo_control(&fo_unit,10, 13);
    
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);
    set_fo_control(&fo_unit, FO_INACTIVE_ADR, 1);
    fo_put_memo("Skip jerk, NI=2, inactive =1 ");
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);
    set_fo_control(&fo_unit, FO_INACTIVE_ADR, (0x3ff<<11)|2|0);
    fo_put_memo("Skip jerk, NI=2, inactive =0 but ignored ");
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);
    set_fo_control(&fo_unit, FO_INACTIVE_ADR, (0x3fe<<11)|2|0);
    fo_put_memo("Skip jerk, NI=2, inactive =0 should work ");
    dump_fo_pipe_port(&fo_unit, &chip);
    dump_fo_host_port(&fo_unit, &chip);

    
    
}
#endif
#ifdef TEST_CALC
void calc_put_memo(char* string)
{
    fprintf(ftestout,"CALC_MEMO:");
    fprintf(ftestout,string);
    fprintf(ftestout,"\n");
}

#define CALC_PHID 42
main()
{
    int i;
    struct chip_control_struct ccu;
    struct calc_struct * calc_unit;
    calc_unit = &(ccu.calcu);
    init_g6control();
    reset_calc_status(calc_unit);
    for(i=0;i<128;i++){
	ccu.local_memory[i] = i*16;
    }
    calc_put_memo("For one particle ");
    set_calc_control(calc_unit, CALC_LRAM_ADR, 2);
    set_calc_control(calc_unit, CALC_LFORCE_ADR, 32);
    set_calc_control(calc_unit, CALC_N_ADR, 1);
    do_calc(&ccu);
    calc_put_memo("For two particle ");
    set_calc_control(calc_unit, CALC_N_ADR, 2);
    do_calc(&ccu);
}
#endif
