/*
 * ECC64.C
 * 64bit ECC generation/error check/correction code
 *
 * Copyright Jun Makino 1998
 *
 * Version 1.0 Feb 15 1998
 *          
 *
 */
#include "grape6sim.h"
#include "ecc64.h"

#define DATABITS 64
#define CHECKBITS 8
#define SIGLEN (DATABITS+CHECKBITS)
#define NPPATTERNS (1<<CHECKBITS)

static int error_location[NPPATTERNS];
static int ecc_status[NPPATTERNS];
static ULONG data_correction[NPPATTERNS];
static ULONG parity_correction[NPPATTERNS];

static int parity_matrix[SIGLEN][CHECKBITS];

static int generation_matrix[DATABITS][CHECKBITS];


int pop_count(int i)
{
  int c, k, l;
  for (k=0, l = 1, c = 0; k<(sizeof(int)*8); k++, l<<=1){
    /*    printf("k, l, c, i = %d 0x%x %d 0x%x\n", k, l, c, i);*/
    if( i & l) c++;
  }
  return c;
}

void set_error_location_table()
{
  int i,j;
  error_location[0] = -2;
  ecc_status[0] = 0;
  for(i=1; i<NPPATTERNS; i++){
    error_location[i] = -1;
    ecc_status[i] = 2;
  }
  for(i=0; i<NPPATTERNS; i++){
    data_correction[i] = ULONG_ZERO;
    parity_correction[i] = ULONG_ZERO;
  }
  for(i=0; i <SIGLEN; i++){
    int k = 0;
    for(j = CHECKBITS-1; j >=0; j--){
      k<<=1;
      k |= parity_matrix[i][j];
    }
    error_location[k] = i;
    ecc_status[k] = 1;
    if (i >= CHECKBITS){
      data_correction[k] = ULONG_ONE <<(i-CHECKBITS);
    }else{
      parity_correction[k] = ULONG_ONE <<i;
    }
  }
  printf("error table\n");
  for(i=0; i<NPPATTERNS; i++){
    printf("%2x %16lx %2lx %x\n", i, data_correction[i], parity_correction[i],
	   ecc_status[i]);
  }
  
}

void set_parity_matrix()
{
  int i, j;
  for(i=0;i<SIGLEN; i++){
    for(j=0;j<CHECKBITS; j++){
      parity_matrix[i][j] = 0;
    }
    parity_matrix[i][CHECKBITS-1] = 1;
  }
  for(i=0;i<CHECKBITS-1; i++){
    parity_matrix[i][i] = 1;
  }
  for(i=1, j = CHECKBITS; j <SIGLEN; j++, i++){
    int k, l;
    while (pop_count(i) <= 1) i++;
    for (k=0, l = 1; k<CHECKBITS-1; k++, l<<=1){
      if( i & l){
	parity_matrix[j][k] = 1;
      }
    }
  }
  for(i=0;i<DATABITS; i++){
    for(j=0; j<CHECKBITS;j++){
      generation_matrix[i][j] = parity_matrix[i+CHECKBITS][j];
    }
    /* last column of G must be adjusted to obtain the reduced form */
    for(j=0; j<CHECKBITS-1;j++){
      generation_matrix[i][CHECKBITS-1] ^= generation_matrix[i][j];
    }
  }
  
  printf("Parity Matrix H \n");
  for(i=0;i<SIGLEN; i++){
    printf("%2x:", i);
    for(j=0;j<CHECKBITS; j++){
      printf(" %1d", parity_matrix[i][j]);
    }
    printf("\n");
  }
  printf("Generation  Matrix G \n");
  for(i=0;i<DATABITS; i++){
    printf("%2x:", i);
    for(j=0;j<CHECKBITS; j++){
      printf(" %1d", generation_matrix[i][j]);
    }
    printf("\n");
  }
  set_error_location_table();

  printf("Parity Matrix H \n");
  for(j=0;j<CHECKBITS; j++){
    for(i=0;i<SIGLEN; i++){
      if( (i%8) == 0)printf(" ");
      printf("%1d", parity_matrix[i][j]);
    }
    printf("\n");
  }
  printf("Generation Matrix G \n");
  for(j=0;j<CHECKBITS; j++){
    for(i=0;i<DATABITS; i++){
      if( (i%8) == 0)printf(" ");
      printf("%1d", generation_matrix[i][j]);
    }
    printf("\n");
  }

}

ULONG generate_parity(ULONG data)
{
  int i,j, k;
  ULONG l;
  int bits[LONGBITS];

  for (k=0, l = 1; k<LONGBITS; k++, l<<=1){
    if( data & l) {
      bits[k] = 1;
    }else{
      bits[k] = 0;
    }
  }
#if 0  
  printf("indata %lx = ", data);
  for (k=LONGBITS-1; k>=0; k--){
    printf("%1d", bits[k]);
  }
  printf("\n");
#endif  
  k = 0;
  for(i = CHECKBITS-1; i >=0; i--){
    k<<=1;
    for(j = 0; j < LONGBITS; j++){
      k ^= bits[j] & generation_matrix[j][i];
    }
  }
#if 0  
  printf("data, parity =  0x%lx, %x\n", data, k);
#endif  
  return (ULONG) k;
}

ULONG check_parity(ULONG data,
		   ULONG parity)
{
  int bits[SIGLEN];
  int i,j, k;
  ULONG l;
  for (k=0, l = 1; k<CHECKBITS; k++, l<<=1){
    if( parity & l) {
      bits[k] = 1;
    }else{
      bits[k] = 0;
    }
  }
  for (k=CHECKBITS, l = 1; k<SIGLEN; k++, l<<=1){
    if( data & l) {
      bits[k] = 1;
    }else{
      bits[k] = 0;
    }
  }
  k = 0;
  for(i = CHECKBITS-1; i >=0; i--){
    k<<=1;
    for(j = 0; j < SIGLEN; j++){
      k ^= bits[j] & parity_matrix[j][i];
    }
  }
#if 0  
  printf("data, parity, result =  0x%lx, %lx %x\n", data, parity, k);
#endif  
  return (ULONG) k;
}

int check_and_correct_data(ULONG *data,
			   ULONG *parity)
     /* return = 1 : single error corrected
	       = 2 : double error uncorrected */
{
  int syndrome;
  syndrome = check_parity(*data, *parity);
  *data ^= data_correction[syndrome];
  *parity ^= parity_correction[syndrome];
  return ecc_status[syndrome];
}  

check_single_correction(ULONG data,
		 int errloc,
		 int perrloc)
{
  ULONG parity, err, perr;
  ULONG newdata, newparity;
  int syndrome, retcode;
  parity = generate_parity(data);
  err = perr = 0;
  if (errloc >= 0) err = ULONG_ONE << (errloc);
  if (perrloc >= 0) perr = ULONG_ONE << (perrloc);
  newdata = data^err;
  newparity = parity^perr;
  retcode = check_and_correct_data(&newdata, &newparity);
  if (( newdata != data) ||( newparity != parity)){
    printf("ERROR: retcode, data, parity = %1d %lx %lx\n",retcode,  data, parity);
  }
}

unsigned int byte_parity(unsigned int x)
{
  unsigned int i, k;
  k = 0;
  for(i = 0; i<8; i++){
    k ^= x & 1;
    x >>=1;
  }
  return k;
}

unsigned int  generate_byte_parity(unsigned int  data)
{
  unsigned int i, k;
  k = 0;
  if (sizeof(unsigned int) != 4){
    printf("int size %d != 4 ... \n", sizeof(unsigned int));
    exit(1);
  }
  for(i=0; i<4; i++){
    k<<=1;
    k |= byte_parity(data >> ((3-i)*8));
  }
  return  k;
}

unsigned int check_byte_parity(unsigned int data,
		      unsigned int parity)
{
  unsigned int new_parity;
  new_parity = generate_byte_parity(data);
  return  parity ^ new_parity;
}

  
  
#ifdef SINGLETEST
main()
{
  ULONG data, parity, err, perr;
  LONG errloc,  perrloc ;
  int syndrome, retcode;
  set_parity_matrix();
  while (1){
    scanf("%lx%ld%ld", &data, &errloc, &perrloc);
    parity = generate_parity(data);
    err = perr = 0;
    if (errloc >= 0) err = ULONG_ONE << (errloc);
    if (perrloc >= 0) perr = ULONG_ONE << (perrloc);
    printf(" data, parity =  %lx %lx\n", data, parity);
    data ^= err;
    parity ^= perr;
    printf("data, parity = %lx %lx\n", data, parity);
    retcode = check_and_correct_data(&data, &parity);
    printf("retcode, data, parity = %1d %lx %lx\n",retcode,  data, parity);
    
    syndrome = check_parity(data, parity);
    printf("err_loc = %d\n", error_location[syndrome]);
    if (syndrome){
      data ^= data_correction[syndrome];
      parity ^= parity_correction[syndrome];
    printf("corrected data, parity, syndrome = %lx %lx %lx\n", data, parity,
	 check_parity(data, parity)  );
    }
  }
}
#endif
#ifdef ZEROTEST
/* test for error vector generation without real data */

main()
{
  ULONG data = 0;
  int errloc,  perrloc ;
  perrloc = -1;
  set_parity_matrix();
  for(errloc = 0; errloc < LONGBITS; errloc ++){
    check_single_correction(data, errloc, perrloc);
  }
  errloc = -1;
  for(perrloc = 0; perrloc < CHECKBITS; perrloc ++){
    check_single_correction(data, errloc, perrloc);
  }
}
#endif
#ifdef SANITYTEST
/* test for reasonably large number of data values  */

main()
{
  ULONG data;
  int errloc=-1, perrloc=-1, i ;
  set_parity_matrix();
  for(data = ULONG_ONE, i = 0; i<100000000; i ++){
      if ( (i % 100001) == 0){
      printf("i,  data = %ld %lx\n",i, data);

    }
    check_single_correction(data, errloc, perrloc);
    data = data * ((LONG) 1234579) +1;
  }
}
#endif
#ifdef TEST
/* test for double error  */

main()
{
  ULONG data, parity;
  int errloc=-1, perrloc=-1 ;
  int error[2];
  int i = 0;
  set_parity_matrix();
  
  for(error[0] = 0; error[0] < SIGLEN; error[0] ++){
    for(error[1] = 0; error[1] < SIGLEN; error[1] ++){
      if (error[0] != error[1]){
	int k;
	int ret;
	data = ULONG_ZERO;
	parity = ULONG_ZERO;
	for(k=0;k<2;k++){
	  if (error[k] < CHECKBITS){
	    parity ^= (ULONG_ONE)<<error[k];
	  }else{
	    data ^= (ULONG_ONE)<<(error[k]-CHECKBITS);
	  }
	}
	ret = check_and_correct_data(&data, &parity);
	if (ret != 2){
	  printf("double error ERROR, errors = %d %d\n", error[0], error[1]);
	  i++;
	}
      }
    }
  }
  printf("Number of errors = %d\n", i);
}
#endif
#ifdef BPTEST
/* test for byte parity  */

main()
{
  unsigned int data, parity;
  while (1){
    scanf("%x", &data);
    printf("data = %x\n", data);
    parity = generate_byte_parity(data);
    printf("data, parity, checkbits = %x %x %x\n", data, parity,
	   check_byte_parity(data, parity));
  }
}
#endif

