/*
 * utils.c
 *
 * utility functions
 *
 * Copyright Jun Makino 1997
 *
 * Version 1.0 Nov 3 1997
 *
 * BIG changes
 * 98/03/01 force_1_round_and_shift now returns the value which
 *             is CHOPPED to out_bits (higher bit masked out )
 * 98/08/03 set_output_file now keeps the table of already opened files
 *          and check if the argument file is in there or not
 *
 */

#include <stdlib.h> 
#include <string.h> 
#include "grape6sim.h"

ULONG shorten_mantissa(ULONG in, LONG bits)
{
  ULONG force = 1;
  int k;
  if((in & ((ULONG_ONE << (bits+1))-1)) == ULONG_ZERO) force = 0;
  return ((in >>bits) | force);
}
    
ULONG force_1_round_and_shift(ULONG s, ULONG inbits, ULONG outbits)
{
    ULONG nshifts, forcebit, mask;
    dprintf(10,"(force_1_round %x, %x, %x\n", (int)s, (int)inbits, (int)outbits);
    if (inbits < outbits){
	/* no rounding necessary */
	return s << (outbits - inbits);
    }else if (inbits == outbits){
	return s;
    }else{
	nshifts = inbits - outbits;
	forcebit = ((ULONG)1)<<nshifts;
	mask = forcebit - 1;
	dprintf(10,"(force_1_round nshifts, mask %x, %x\n", (int)nshifts, (int)mask);
	if (s & mask){ /* non-zero LSBs, need force-1*/
	    s |= forcebit;
	}
	return ((s >> nshifts) & (((ULONG_ONE)<<outbits)-1));
    }
}

ULONG force_1_round_and_shift_grape_float(ULONG in,
					  ULONG inbits,
					  ULONG outbits)
{
  ULONG exp1, sign1, zero1, mantissa1;
  ULONG mantissa2;
  decompose_float(in, inbits, &exp1, &sign1, &zero1, &mantissa1);
  mantissa2 = force_1_round_and_shift(mantissa1, inbits, outbits);
  return compose_float(outbits,  exp1, sign1, zero1, mantissa2);
}



void print_in_binary(ULONG num, ULONG nbits)
{
    int i;
    for(i=nbits-1;i>= 0;i--){
        printf("%1d",(num >> i) & 1);
    }
}


void decompose_float(ULONG data, ULONG mantissa_bits, ULONG *exponent,
		     ULONG * sign, ULONG * zero, ULONG * mantissa)
{
  *mantissa = data & (((ULONG) 1)<<mantissa_bits)-1;
  *zero = (data >>(mantissa_bits))&1;
  *sign = (data >>(mantissa_bits+1))&1;
  *exponent = (data >>(mantissa_bits+2)) & EXP_MASK;
}

ULONG compose_float(ULONG mantissa_bits, ULONG exponent,
		     ULONG  sign, ULONG  zero, ULONG  mantissa)
{
  ULONG data;
  exponent &= (ULONG) EXP_MASK;
  sign &=  (ULONG)ULONG_ONE;
  zero &=  (ULONG)ULONG_ONE;
  mantissa &= (ULONG_ONE<<mantissa_bits)-1;
  data =  (exponent<<(mantissa_bits + 2))
    | (sign<<(mantissa_bits + 1))
    | (zero<<mantissa_bits)
    | mantissa;
  dprintf(9,"(compose_float) exp, sign, zero, man = %x %x %x %x ",
	  (int)exponent, (int)sign, (int)zero, (int)mantissa);
#ifdef X86
  dprintf(9," data =  %Lx\n", data);
#else
  dprintf(9," data =  %lx\n", data);
#endif  
  return data;
  
}


double convert_grape_float_to_double(ULONG indata, ULONG inbits)
{
  ULONG sign, exponent, mantissa, zero;
  double x;
  decompose_float(indata, inbits, &exponent, &sign, &zero, &mantissa);
  x = mantissa;
  x=ldexp(x, ((int)exponent) - INTERACTION_POSITION_EXP_OFFSET - inbits);
  if(zero) x = 0;
  if(sign) x = -x;
  return x;
}

double convert_grape_fixed_to_double(LONG indata, ULONG inbits, LONG offset)
{
  if(inbits < ((ULONG)LONGBITS)){
    /* need to extend sign before conversion */
    if (indata>>(inbits-1)){
      /* negative sign ... */
      LONG sgn_mask = - ((((LONG) 1)<<((int)inbits)-1));
      dprintf(2,"convert_grape_fixed, indata, inbits, mask = %lx %lx %lx\n",
	      indata, inbits, sgn_mask);
      indata |= sgn_mask;
    }
  
  }
  dprintf(2,"convert_grape_fixed, indata, offset = %lx %lx %lx\n",
	      indata, offset);
  return CONVERT_GRAPE_INT_POS_TO_DOUBLE(indata,(int)offset);
}

ULONG convert_double_to_grape_float_bitwise(double x, ULONG float_bits)
{
  ULONG sign, exponent, mantissa, zero;
  ULONG exp2, ux;
  double xorg;
  int iexp;
  ux = (*((ULONG*)(&x)));
  xorg = x;
  sign = zero = 0;
  if (x < 0.0){
    sign = 1;
    x = -x;
  }else if(x == 0.0){
    zero = 1;
  }
  
  dprintf(3,"convert_double x = %.20e %ld\n", x, float_bits);
  x = frexp(x, &iexp);
  dprintf(3,"convert_double matissa = %.20e \n", x);
  mantissa = rint(ldexp(x, (int)float_bits));
  if (mantissa >>float_bits){
    iexp ++;
    mantissa >>=1;
  }
  dprintf(3,"convert_double matissa = %.20e \n", x);
  exponent = iexp + INTERACTION_POSITION_EXP_OFFSET;
  exp2 =  (ux>>52) &0x7ffL;
  exp2 = ((exp2 & 0x400L) >>1) | (exp2 & 0x1ffL) + 2;
  dprintf(3,"exp, exp2 = %ld, %ld\n", exponent, exp2);
  return compose_float(float_bits, exp2, sign, zero, mantissa);
}
ULONG convert_double_to_grape_float(double x, ULONG float_bits)
{
  ULONG sign, exponent, mantissa, zero;
  int iexp;
  sign = zero = 0;
  if (x < 0.0){
    sign = 1;
    x = -x;
  }else if(x == 0.0){
    zero = 1;
  }
  dprintf(3,"convert_double x = %.20e %ld\n", x, float_bits);
  x = frexp(x, &iexp);
  dprintf(3,"convert_double matissa = %.20e \n", x);
  mantissa = rint(ldexp(x, (int)float_bits));
  if (mantissa >>float_bits){
    iexp ++;
    mantissa >>=1;
  }
  dprintf(3,"convert_double matissa = %.20e \n", x);
  exponent = iexp + INTERACTION_POSITION_EXP_OFFSET;
  if (zero == 0){
      return compose_float(float_bits, exponent, sign, zero, mantissa);
  }else{
      return 0x609800000L;
  }
}


ULONG convert_float_to_grape_float(float x, ULONG float_bits)
{
  ULONG sign, exponent, mantissa, zero;
  ULONG exp2, g1, g2;
  float xorg;
  unsigned int ux;
  ULONG ulx;
  int iexp;
  if (float_bits != 24L){
      fprintf(stderr,"convert_float_to_grape_float_bitwise can't be used for");
      fprintf(stderr,"mantissa length = %ld\n", float_bits);
      exit(-1);
  }
  xorg = x;
  ux = (*((unsigned int*)(&xorg)));
  ulx = ux;
  zero = 0;
  sign = ulx>>31;
  if(x == 0.0){
      zero = 1;
  }
  exponent = (ulx>>23) & 0xffL;
  mantissa = (1L<<23) | (ulx & 0x7fffffL);
  exponent = (exponent & 0xffL)+386;
  g1 = compose_float(float_bits, exponent, sign, zero, mantissa);
  g2 = convert_double_to_grape_float((double) x, float_bits);
  if (g1 != g2){
      fprintf(stderr,"differ: %g, %lx %lx\n", x, g1, g2);
  }
  return g1;
}

ULONG priority_encoder(ULONG x)
{
  int i;
  dprintf(3,"priority encoder, size = %d %d\n", sizeof(ULONG), sizeof(LONG));
  for(i=LONGBITS-1; i>=0;i--){
    dprintf(11,"i, x, mask = %d, %lx, %lx\n", i, x, ((LONG)1)<<i);
    dprintf(11,"i, x, mask = %d, %8x%8x, %8x%8x\n", i, (unsigned int) (x>>32),(unsigned int)(x&0xffffffff),
	    (unsigned int)((((LONG)1)<<i) >>32),  (unsigned int)((((LONG)1)<<i) &0xffffffff));
    if (x & (((LONG)1)<<i)){
      return i;
    }
  }
  return 0;
}

ULONG convert_fixed_to_grape_float(ULONG in, ULONG in_flac_len, ULONG float_bits)
{
  ULONG sign, exponent, mantissa, zero;
  int iexp, loc;
  loc = (int) priority_encoder(in)+1; 
  iexp = loc - in_flac_len;
  sign = zero = 0;
  if(in == 0){
    zero = 1;
  }
  mantissa = force_1_round_and_shift(in, loc, float_bits);
  exponent = iexp + INTERACTION_POSITION_EXP_OFFSET;
  return compose_float(float_bits, exponent, sign, zero, mantissa);
}

ULONG compare_grape_floats(ULONG in1, ULONG in2, ULONG inbits)
     /* return 1 if in1 > in2
	       0 otherwize */
     
{
  ULONG sign1, exponent1, mantissa1, zero1;
  ULONG sign2, exponent2, mantissa2, zero2;
  ULONG result;
  decompose_float(in1, inbits, &exponent1, &sign1, &zero1, &mantissa1);
  decompose_float(in2, inbits, &exponent2, &sign2, &zero2, &mantissa2);
  if (zero1 == 1) {
    result = 0;
  }else if (zero2 == 1){
    result = 1;
  }else if (exponent1 > exponent2){
    result = 1;
  }else if (exponent1 == exponent2){
    if (mantissa1 > mantissa2) {
      result = 1;
    }else{
      result = 0;
    }
  }else{
    result = 0;
  }
  return result;
}


/*
 * Get data file name etc from environment
 *
 * env names and defaults
 * Name                  default
 * G6SIM_PIPE_OUT_FILE   stdout
 */

#define MAXOUTFILE 100
#define MAXOUTFNAME 512
void set_output_file(FILE  **fout, char * name)
{
    char * p;
    static FILE * fptrs[MAXOUTFILE];
    static char   fnames[MAXOUTFILE][MAXOUTFNAME];
    static int fused = 0;
    int i;
    p = getenv(name);
    if(p == NULL) {
	*fout = stdout;
	fprintf(stderr, "(set_output file) use stdout for %s\n", name);
    }else{
	/* first check if the file is already open or not */
	for (i=0;i<fused; i++){
	    printf("set_output_file, checking file %d %s %s\n",
		   i, fnames[i], p);
	    if (strcmp(p,fnames[i]) == 0){
		*fout = fptrs[i];
		fprintf(stderr, "(set_output file) reuse file %s for %s\n",p, name);
		return;
	    }
	}
	/* file is new. Create one and register it */
	*fout = fopen(p, "w");
	if ( *fout == NULL){
	    fprintf(stderr,"(set_output_file) failed to open output file  %s\n",
		    name);
	    exit(-1);
	}
	fprintf(stderr, "(set_output file) use file %s for %s\n",p, name);
	fptrs[fused] = *fout;
	strcpy(fnames[fused],p);
	fused ++;
    }
}


/*
 * DPRINTF.C: print debug messages, if requested debug level high enough
 *
 * xx-Mar-88  Peter Teuben: created
 * 14-Jun-88  Josh Barnes: moved to seperate source file.
 * 10-sep-90  PJT: routines to set_ and get_ the debug level
 *  8-jul-91  pjt: using VFPRINTF now - gcc should work again
 * 12-oct-91  pjt: added __BORLANDC__ to vfprintf users
 * 25-feb-92  pjt: happy gcc2.0
 *  6-apr-93  pjt: took debug_level back into this routine
 * 21-sep-93  pjt: ANSI
 *
 *	GCC complains about va_start() being not OK
 *	GCC v1.39 stopped complaining.... but actually doesn't work
 */


static int debug_level=0;

void set_debug_level(int level)
{
  debug_level = level;
}


void set_debug_level_from_env()
{
    char * p;
    p = getenv("G6SIM_DEBUG_LEVEL");
    if(p != NULL) {
	int level;
	level = atoi(p);
	fprintf(stderr,"set_debug_level to %d (%s)\n", level, p);
	debug_level = level;
    }
}


#if defined(__STDC__) || defined(__cplusplus)
/* Posix compliant */
#include <stdarg.h>


/*
 * DPRINTF: printf-style debugging messages, controlled by debug_level
 *	    as set by the user interface (debug=)
 */

void dprintf(int debug, char *  fmt, ...)
{
    va_list ap;


    if (debug <= debug_level) {		/* print this debug message? */
        va_start(ap, fmt);	
#if defined(NEED_DOPRNT)
	_doprnt(fmt, ap, stderr);	/*   use low-level interface */
#else
	vfprintf(stdout, fmt, ap);	/* this should be ok on sun for now */
#endif
					/*   may become vprintf in ANSI C */
	fflush(stderr);			/*   drain std error buffer */
        va_end(ap);
    }
}
#else

/* Old K&R version */

#include <varargs.h>

/*VARARGS1*/
void dprintf(debug, va_alist)
int debug;
va_dcl
{
    va_list l;
    string fmt;

    va_start(l);			/* l starts with string */
    fmt = va_arg(l, string);		/* point to string */ /*PPAP*/
					/* l now has remainder */	
    if (debug <= debug_level) {		/* print this debug message? */
#if defined(NEED_DOPRNT)
	_doprnt(fmt, l, stderr);	/*   use low-level interface */
#else
	vfprintf(stderr, fmt, l);	/* this should be ok on sun for now */
#endif
					/*   may become vprintf in ANSI C */
	fflush(stderr);			/*   drain std error buffer */
    }
    va_end(l);
}

#endif

#ifdef TEST
main()
{
    ULONG g6data;
    double dx, dx2;
    float rx, rx2;
    set_debug_level(10);
    while (1){
	printf("enter x: ");
	scanf("%le",&dx);
	rx = dx;
	printf("x, g6data, rx = %le %lx %lx %lx %lx\n",
	       dx, convert_double_to_grape_float_bitwise(dx, 24),
	       convert_float_to_grape_float((float)dx, 24),
	       *((int*)(&rx)),*((long*)(&dx)));
	rx2 = convert_grape_float_to_double(convert_float_to_grape_float((float)dx, 24), 24);
	printf("rx, rx2 = %x %x\n",
	       *((int*)(&rx)), *((int*)(&rx2)));
	
    }
}
#endif


