/*
 * CUTOFF_FUNCTIONS.C
 *
 * generic power law interploator
 *
 * function power_function
 *
 * Copyright Jun Makino 1997
 *
 * Version 1.0 Dec 6 1997
 *
 * 
 *
 */
#include "grape6sim.h"

#define TABLE_SIZE  CUTOFF_TABLE_SIZE 
#define TABLE_BITS CUTOFF_TABLE_BITS 
#define NUMBER_OF_TABLES CUTOFF_NUMBER_OF_TABLES 
static long table_entry_bits = TABLE_BITS;
static long int zeroth_tab[NUMBER_OF_TABLES][TABLE_SIZE];
static long int first_tab[NUMBER_OF_TABLES][TABLE_SIZE];
static long int first_sign[NUMBER_OF_TABLES][TABLE_SIZE];

void dump_cutoff_table()
{
  int i,j;
  for(i=0;i<TABLE_SIZE; i++){
    printf("CUT %2lx", i);
    for(j=0;j<2;j++){
      printf(" %4lx %4lx %1lx", zeroth_tab[j][i],first_tab[j][i],
	     first_sign[j][i]);
    }
    printf("\n");
  }
}
  
void print_cutoff_table(FILE * fout)
{
  int i,j;
  for(i=0;i<TABLE_SIZE; i++){
    fprintf(fout,"CUT %2lx", i);
    for(j=0;j<2;j++){
      fprintf(fout, " %4lx %4lx %1lx", zeroth_tab[j][i],first_tab[j][i],
	     first_sign[j][i]);
    }
    fprintf(fout,"\n");
  }
}
  
ULONG cutoff_table_data(int tabid,/* 0: zeroth, 1: first, 2: sign */
			   int tabno,
			int address)

{
    long int retval;
    if (tabid == 0){
	retval = zeroth_tab[tabno][address];
    }else if (tabid == 1){
	retval = first_tab[tabno][address];
    }else if (tabid == 2){
	retval =  first_sign[tabno][address];
    }
    return (ULONG) retval;
}

  
void set_cutoff_table_data(int data,
			   int tabid,/* 0: zeroth, 1: first, 2: sign */
			   int tabno,
			   int address)
     
{
    long int retval;
    if (tabid == 0){
	zeroth_tab[tabno][address] = data;
    }else if (tabid == 1){
	first_tab[tabno][address] = data;
    }else if (tabid == 2){
	first_sign[tabno][address] = data;
    }
}

static pcutoff cutoff_function_table[NUMBER_OF_TABLES];

double samplefunction(double x);

void prepare_cutoff_table(ULONG table_id,
		    ULONG table_size,
		    ULONG zeroth_len,
		    pcutoff cutoff)
     /* I guess this works only for table_size = 64 */

{
  double x, f, df, d1f, dx;
  double pindex, f0, df0, d1f0, fscale;
  int if0, idf0, id1f0, half_table,i;
  cutoff_function_table[table_id] = cutoff;
  dx = 1.0/table_size;
  half_table = (int) table_size/2;
  for (i=0; i<((int)table_size); i++){
    double f1, fhalf;
    x = i*dx;
    f = (*cutoff)(x);
    f1 = (*cutoff)(x+dx);
    fhalf = (*cutoff)(x+dx*0.5);
    df = (f1-f)/dx;
    f0 = f+0.5*(fhalf - 0.5*(f+f1));
    if (f0 > 1.0) f0 = 1;
    /*    f0 = f;*/
    df *= dx;
    dprintf(10, "prepare_tab  x,  f, f1, fhalf, f0 =  %le  %le %le  %le %le\n",
	    x,f,f1,fhalf, f0);
    
    fscale = ULONG_ONE<<((int) zeroth_len );
    zeroth_tab[table_id][i] = rint(f0*fscale);
    idf0 = rint(df*fscale);
    if (idf0 > 0){
      first_tab[table_id][i] = idf0;
      first_sign[table_id][i] = 0;
    }else{
      first_tab[table_id][i] = -idf0;
      first_sign[table_id][i] = 1;
    }
      
    dprintf(6, "prepare_tab i, x, dx,  f, df, df1 = %4d %le %le  %le %le %lx\n",
	    i, x, dx,  f0, df,first_tab[table_id][i]);
  }
}
    
      
ULONG cutoff_function( ULONG in, /* multiplicant */
		      ULONG inbits, /* word length for input 1 */
		       ULONG table_id)
{
  ULONG table_index;
  ULONG table_msb;
  ULONG fsign;
  LONG zeroth, first,  remaindar, remaindar_mask;
  ULONG inmax = (1<<inbits)-1;
  int first_out_shifts = (int)inbits - table_entry_bits;
  if (in >inmax) in = inmax;
  table_index = in >> ((int)inbits - table_entry_bits);
#ifdef X86
  dprintf(1, "in, inb, table_en = %Lx %Lx %x\n", in, inbits, table_entry_bits);
  dprintf(1, "orig. table_index =  %Lx\n", table_index);
#else
  dprintf(1, "in, inb, table_en = %lx %lx %x\n", in, inbits, table_entry_bits);
  dprintf(1, "orig. table_index =  %lx\n", table_index);
#endif  
  remaindar_mask =  ((ULONG_ONE << ((int)inbits - table_entry_bits))-1);
  remaindar = in & remaindar_mask;
  dprintf(1, "table, remaindar = %x %x %x\n", (int) table_index, (int) remaindar, (int) remaindar_mask);
  zeroth = zeroth_tab[table_id][table_index];
  first = first_tab[table_id][table_index];
  fsign = first_sign[table_id][table_index];
  dprintf(1, " 0th, 1st = %lx %lx \n",  zeroth, first);
  if (fsign){
    zeroth -=  ((first*remaindar)>>first_out_shifts);
  }else{
    zeroth +=  ((first*remaindar)>>first_out_shifts);
  }
  dprintf(1, "(cutoff_function) returns %lx \n",  zeroth);
  return zeroth;
}

ULONG cutoff_in_grape_float( ULONG in, /* input distance  */
			     ULONG inscale, /* input inverse scale length */
			     ULONG inbits,  /* input mantissa length */
			     ULONG outbits, /* output mantissa length */
			     ULONG table_id) /* function table index */
     /* accepts inputs and gives the result in GRAPE-6 float format */
{
    ULONG exponent, sign, zero, mantissa, in_fixed, cutoff_fixed;
    int real_exp;
    dprintf(2,"(cutoff) in=      %x %le\n",(int)in,convert_grape_float_to_double(in,CUTOFF_MANTISSA_LEN));
    dprintf(2,"(cutoff) inscale= %x %le\n",(int)inscale,convert_grape_float_to_double(inscale,CUTOFF_MANTISSA_LEN));    
  /* first, multiply in and inscale */
  
  in = mult(in, inscale, CUTOFF_MANTISSA_LEN,
	    CUTOFF_MANTISSA_LEN,CUTOFF_MANTISSA_LEN);
  /* then, convert it to fixed point */
    decompose_float(in, inbits, &exponent, &sign, &zero, &mantissa);
    dprintf(6,"(cutoff) mult result in hex =%d %d %d %x\n",exponent, sign, zero, mantissa);    
    real_exp = ((int)exponent) - INTERACTION_POSITION_EXP_OFFSET;
    if (real_exp > 0){
      in_fixed = (ULONG_ONE <<inbits)-1;
    }else if(real_exp < - ((int) inbits)){
      in_fixed = 0;
    }else{
      in_fixed = mantissa >>(-real_exp);
    }
    if(zero) in_fixed = 0;
    cutoff_fixed =  cutoff_function( in_fixed,  inbits, table_id);
    return convert_fixed_to_grape_float(cutoff_fixed,CUTOFF_MANTISSA_LEN ,
					outbits);
}


  

double samplefunction(double x)
{
  return exp(-x*x*16);
}

double linear_cuttoff(double x);
double linear_cuttoff(double x)
{
  return x;
}


double p0(double x);
double f0(double x);

double pscaled(double x)
{
  return p0(x*4);
}

double fscaled(double x)
{
  return f0(x*4);
}


double nocut(double x)
{
  return (double)1.0;
}



void set_gaussian_cutoff()
{
  prepare_cutoff_table((ULONG)0, (ULONG)TABLE_SIZE,  CUTOFF_MANTISSA_LEN,
		       &pscaled);
  prepare_cutoff_table((ULONG)1, (ULONG)TABLE_SIZE,  CUTOFF_MANTISSA_LEN,
		       &fscaled);
}

void reset_cutoff()
{
  prepare_cutoff_table((ULONG)0, (ULONG)TABLE_SIZE,  CUTOFF_MANTISSA_LEN,
		       &nocut);
  prepare_cutoff_table((ULONG)1, (ULONG)TABLE_SIZE,  CUTOFF_MANTISSA_LEN,
		       &nocut);
}

void set_linear_cutoff()
{
  prepare_cutoff_table((ULONG)0, (ULONG)TABLE_SIZE,  CUTOFF_MANTISSA_LEN,
		       &linear_cuttoff);
  prepare_cutoff_table((ULONG)1, (ULONG)TABLE_SIZE,  CUTOFF_MANTISSA_LEN,
		       &linear_cuttoff);
}

double original_cutoff(double x, ULONG table_id)
{
  return (*cutoff_function_table[table_id])(x);
}


double cutoff_in_double(double x, ULONG tableid, ULONG zeroth_len);

double cutoff_in_double(double x, ULONG tableid, ULONG zeroth_len)
{
  ULONG ix, result;
  int nshift = zeroth_len ;
  ix = x * (1<<nshift);
  result = cutoff_function(ix,  zeroth_len, tableid);
  return ((double)result)/(1<<(nshift));
}
double cutoff_in_double_through_grape(double x, ULONG tableid, ULONG in_len,
				      ULONG out_len)
{
  ULONG in,inscale, result;
  in= convert_double_to_grape_float(x, in_len);
  inscale =  convert_double_to_grape_float((double) 1, in_len);
  result = cutoff_in_grape_float(in, inscale , in_len, out_len,tableid);
  return convert_grape_float_to_double(result, out_len);
}

#ifdef TEST
main()
{
  ULONG   zeroth_len, table_size, nshift;
  double x, exact, gresult, err;
  ULONG ix, result;
  set_debug_level(10);
  printf(" zeroth_len, table_bits ");
  scanf("%ld%ld", &zeroth_len, &table_entry_bits);
  nshift = zeroth_len - 1;
  table_size = ULONG_ONE << table_entry_bits;
  prepare_cutoff_table(0, table_size,  zeroth_len, &pscaled);
  prepare_cutoff_table(1, table_size,  zeroth_len, &fscaled);
  dump_cutoff_table();
  printf("enter x (-1 for end): ");
  scanf("%le",&x);
  while(x >= 0){
    double gresult2 =  cutoff_in_double_through_grape(x, 0,
						      zeroth_len, (ULONG) 24);
    gresult = cutoff_in_double(x, 0,  zeroth_len);
    exact = pscaled(x);
    err = exact - gresult;
    if(exact != 0.0){
      err /= exact;
    }
    printf(" result = 0x%lx %le %le %le %le\n",
	   result, exact, gresult, gresult2, err);
    printf("enter x (-1 for end): ");
    scanf("%le",&x);
  }
}

#endif

#ifdef XTEST
main()
{
  ULONG   zeroth_len, table_size;
  double x, exact, gresult, err, dx;
  ULONG ix, result;
  int i, n, nshift;
  set_debug_level(0);
  fprintf(stderr," zeroth_len, table_bits ");
  scanf("%ld%ld", &zeroth_len, &table_entry_bits);
  nshift = zeroth_len - 1;
  table_size = ULONG_ONE << table_entry_bits;
  prepare_cutoff_table((ULONG)0, table_size,  zeroth_len, &pscaled);
  prepare_cutoff_table((ULONG)1, table_size,  zeroth_len, &fscaled);
  dump_cutoff_table();
  n = 1<<nshift;
  dx = 0.5/n;
  n*= 2;
  for(i=0;i<=n; i++){
    ULONG j;
    x = i*dx;
    printf("%.6lf ", x);
    for(j=0;j<2;j++){
      gresult = cutoff_in_double(x, j,  zeroth_len);
      if(j==0){
	exact = pscaled(x);
      }else{
	exact = fscaled(x);
      }
      err = exact - gresult;
      printf("%le %le %le ",
	     exact, gresult, err);
    }
    printf("\n");
  }
}

#endif

