/*
 * Copyright (c) 1997 Picture Elements, Inc.
 *    Stephen Williams (steve@picturel.com)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version. In order to redistribute the software in
 *    binary form, you will need a Picture Elements Binary Software
 *    License.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *  ---
 *    You should also have recieved a copy of the Picture Elements
 *    Binary Software License offer along with the source. This offer
 *    allows you to obtain the right to redistribute the software in
 *    binary (compiled) form. If you have not received it, contact
 *    Picture Elements, Inc., 777 Panoramic Way, Berkeley, CA 94704.
 */
#ident "$Id:$"

/*
 * This is an example driver that supports memory mapping.
 *
 * Memory mapping in the Linux kernel operates around the
 * vm_area_struct, which represents a contiguous stretch of address
 * space with identical properties. The vm_area_struct represents a
 * virtual segment, which may or may not be in physical memory.
 *
 * The kernel generally assumes that pages are initially NOT in
 * physical memory, and faults the page in as needed. The
 * vm_area_struct for the segment contains pointers to the operations
 * needed to manipulate the pages of the segment.
 *
 * This example driver maintains a virtual frame buffer in
 * memory. Pages are created and zero-fixxed as they are initially
 * accessed, but then remain until the driver is unloaded.
 */

# include  <linux/module.h>
# include  <linux/modversions.h>
# include  <linux/version.h>
# include  <linux/wrapper.h>
# include  <linux/fs.h>
# include  <linux/mm.h>
# include  <asm/io.h>

# define NUM_TEST 1
# define TEST_PAGES 4

struct Instance {
      unsigned long pages[TEST_PAGES];
};

struct Instance inst [NUM_TEST];

/*
 * Normally, when a process uses mmap to map a stretch of memory, the
 * xxmmap function is called. However, there are few occasions where
 * the kernel creates new mappings spontaneously. Examples are:
 *
 *    When forking, mmapped regions are duplicated by creating
 *    duplicate vm_area_structs for the child.
 *
 *    If this application unmaps a hole in the mapped region, the
 *    kernel will create a new vm_area_struct to represent the new
 *    smaller segment.
 *
 * In any case, dentries, inodes, vm_ops, etc. are all taken care
 * of. Thus, the most common use of this function is to help with
 * reference counting.
 */
static void xxopen(struct vm_area_struct*vma)
{
      printk("xxopen\n");

	/* A new reference was spontaneously created. Count it. */
      MOD_INC_USE_COUNT;
}

/*
 * When a vm_area_struct is removed, no matter how it is created, this
 * method is called. The kernel will take care of things like
 * releasing the dentry/inode. This is really only a hook to manage
 * reference counting.
 */
static void xxclose(struct vm_area_struct*vma)
{
      printk("xxclose\n");
      MOD_DEC_USE_COUNT;
}

/*
 * When the process unmaps some memory in this vm_area_struct, the
 * kernel calls this function. If you manage individual pages, you can
 * use this to keep track of individual pages that are no longer
 * needed.
 *
 * Note that there may be more then one unmap before close is actually
 * called. Also, unmap may unmap only portions of the mapped area.
 */
static void xxunmap(struct vm_area_struct *area, unsigned long off,
		    size_t size)
{
      printk("xxunmap at address=%p, size=%p\n", off, size);
}

/*
 * This function is the workhorse of the memory mapped module. The
 * memory mapping created the memory region, but did not necessarily
 * attach the individual page. This function is called by the kernel
 * when a process that mapped the module page-faults on some page in
 * the region. If the access is requesting write access, then the
 * write_access flag will be true.
 *
 * If the page is always in memory, for example a frame buffer, then
 * this function is pretty trivial: return the *physical* address of the
 * page.
 *
 * If the page is not necessarily in memory, then this function
 * arranges for a copy to appear in some page and returns that copy.
 */
static unsigned long xxnopage(struct vm_area_struct*vma,
			      unsigned long address,
			      int write_access)
{
      unsigned long page_nr;

#if LINUX_VERSION_CODE > 0x020100
      struct inode*inode = vma->vm_dentry->d_inode;
#else
      struct inode*inode = vma->vm_inode;
#endif
      unsigned minor = MINOR(inode->i_rdev);
      struct Instance*xsp = inst+minor;

      printk("TEST: page fault at addres=%p vma=%p (start=%p)\n", address,
	     vma, vma->vm_start);

      page_nr = (address - vma->vm_start + vma->vm_offset) / PAGE_SIZE;


	/* If the page was already allocated, then use it as the
	   return value. Otherwise, I need to allocate the page. In a
	   device such as a frame buffer, I would just return the
	   physical address here. */

      if (xsp->pages[page_nr]) {
	    printk("TEST: Select page number %u of device %u flags=%x\n",
		   page_nr, minor, mem_map[MAP_NR(xsp->pages[page_nr])].flags);
	    return xsp->pages[page_nr];
      }

      printk("TEST: Allocating page number %u of device %u\n", page_nr, minor);

      xsp->pages[page_nr] = get_free_page(GFP_KERNEL);

	/* This prevents anything ever happening to the
	   page. Reserving it essentially removes it from any of the
	   kernel's attentions. This makes it permanent in memory. */

      mem_map_reserve(MAP_NR(xsp->pages[page_nr]));

      return xsp->pages[page_nr];
}

static const struct vm_operations_struct vm_ops = {
      xxopen,
      xxclose,
      xxunmap,
      0,
      0,
      0,
      xxnopage,
      0,
      0,
      0
};

/*
 * When a memory map is performed, the linux kernel creates and
 * initializes a vm_area_struct that describes the area. The mmap
 * entry point then does what is needed to connect that struct to the
 * file. It also checks that the ranges are correct. It does not need
 * to perform the actual mapping. Instead, the pages of the area are
 * faulted in with the xxnopage function.
 */
static int xxmmap(struct inode*ino,
		  struct file*file,
		  struct vm_area_struct*vma)
{
	/* This prevents the kernel attempting to swap the
	   segment. There is no point because the pages are all
	   reserved. */
      vma->vm_flags |= VM_LOCKED;

	/* It is up to the mmap entry point to make sure the region
	   requested is within the confines of the device being
	   mapped. */
      if ((vma->vm_offset + (vma->vm_end-vma->vm_start)) >=
	  (TEST_PAGES*PAGE_SIZE)) return -EINVAL;
      
	/* This is what turns the vm_area_struct in to a mapped device
	   of my kind. */
      vma->vm_ops = &vm_ops;

#if LINUX_VERSION_CODE > 0x020100
	/* Put into the vm_dentry the reference counted dcache
	   pointer. This logically attaches the device that I am to
	   the vm_area_struct. */
      vma->vm_dentry = dget(file->f_dentry);
#else
	/* Older kernels (2.0 and earlier) attached inodes to the
	   vm_area_struct because dentrys were not yet
	   invented. Rember to reference-count the inode. */
      vma->vm_inode = ino;
      ino->i_count++;
#endif

	/* A process may close the device after mapping it and still
	   expect the device to behave correctly in the memory
	   region. Thus, a map must be included in the module
	   reference count. */
      MOD_INC_USE_COUNT;

      return 0;
}

static const struct file_operations ops = {
      0,
      0,
      0,
      0,
      0,
      0,
      xxmmap,
      0,
      0,
      0,
      0,
      0,
      0
};

int init_module(void)
{
      unsigned idx;
      for (idx = 0 ;  idx < NUM_TEST ;  idx += 1) {
	    struct Instance*xsp = inst+idx;
	    unsigned pag;

	    for (pag = 0 ;  pag < TEST_PAGES ;  pag += 1)
		  xsp->pages[pag] = 0;
      }

      register_chrdev(40, "test", &ops);
      return 0;
}

void cleanup_module(void)
{
      unsigned idx;

      unregister_chrdev(40, "test");

      for (idx = 0 ;  idx < NUM_TEST ;  idx += 1) {
	    struct Instance*xsp = inst+idx;
	    unsigned page_nr;

	      /* Release all the pages. Remember to unreserve them as
		 well, or the free_page() will not return the page to
		 the allocation pool. */

	    for (page_nr = 0 ;  page_nr< TEST_PAGES ;  page_nr += 1)
		  if (xsp->pages[page_nr]) {
			printk("TEST: cleanup page %u", page_nr);
			mem_map_unreserve(MAP_NR(xsp->pages[page_nr]));
			free_page(xsp->pages[page_nr]);
		  }
      }
}

/*
 * $Log: $
 */

